<?php
use yii\widgets\ActiveForm;
use yii\helpers\Html;
use yii\widgets\MaskedInput;

$this->title = 'Регистрация';
?>

<style>
    :root {
        --primary-green: #4CAF7D;
        --light-green: #E8F5E9;
        --accent-green: #81C784;
        --dark-green: #388E3C;
        --white: #FFFFFF;
        --gray-light: #F5F5F5;
        --gray: #757575;
        --gray-dark: #424242;
        --warning: #FF9800;
        --error: #f44336;
    }

    .register-page {
        min-height: 100vh;
        background: linear-gradient(135deg, var(--light-green) 0%, var(--white) 100%);
        padding: 40px 0;
        position: relative;
        overflow: hidden;
    }

    /* Анимированный фон */
    .register-background {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        z-index: 0;
        opacity: 0.1;
    }

    .register-shape {
        position: absolute;
        border-radius: 50%;
        background: var(--primary-green);
        filter: blur(60px);
        opacity: 0.3;
        animation: float 20s infinite ease-in-out;
    }

    .register-shape:nth-child(1) {
        width: 400px;
        height: 400px;
        top: -100px;
        right: -100px;
        animation-delay: 0s;
    }

    .register-shape:nth-child(2) {
        width: 300px;
        height: 300px;
        bottom: -50px;
        left: -50px;
        animation-delay: -5s;
    }

    .register-shape:nth-child(3) {
        width: 200px;
        height: 200px;
        top: 50%;
        left: 10%;
        animation-delay: -10s;
    }

    @keyframes float {
        0%, 100% {
            transform: translate(0, 0) rotate(0deg);
        }
        33% {
            transform: translate(30px, 30px) rotate(120deg);
        }
        66% {
            transform: translate(-20px, 40px) rotate(240deg);
        }
    }

    /* Основной контейнер */
    .register-container {
        position: relative;
        z-index: 1;
        max-width: 600px;
        margin: 0 auto;
        padding: 0 20px;
    }

    /* Заголовок */
    .register-header {
        text-align: center;
        margin-bottom: 50px;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease forwards;
    }

    .register-title {
        font-size: 3rem;
        font-weight: 800;
        background: linear-gradient(45deg, var(--dark-green), var(--primary-green));
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        margin-bottom: 15px;
    }

    .register-subtitle {
        font-size: 1.2rem;
        color: var(--gray);
        max-width: 600px;
        margin: 0 auto;
    }

    /* Форма */
    .register-form-section {
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.3s forwards;
    }

    .form-card {
        background: var(--white);
        border-radius: 20px;
        padding: 40px;
        box-shadow: 0 15px 40px rgba(76, 175, 125, 0.1);
    }

    /* Поля формы */
    .form-group {
        margin-bottom: 25px;
        position: relative;
    }

    .form-label {
        display: block;
        margin-bottom: 8px;
        color: var(--dark-green);
        font-weight: 600;
        font-size: 1rem;
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .form-label::after {
        content: '*';
        color: var(--error);
        font-size: 1.2rem;
        opacity: 0.7;
    }

    .form-label.optional::after {
        content: '';
    }

    .form-input {
        width: 100%;
        padding: 16px 20px;
        border: 2px solid var(--light-green);
        border-radius: 12px;
        font-size: 1rem;
        color: var(--gray-dark);
        background: var(--white);
        transition: all 0.3s ease;
        outline: none;
        font-family: inherit;
    }

    .form-input:focus {
        border-color: var(--primary-green);
        box-shadow: 0 0 0 4px rgba(76, 175, 125, 0.1);
        transform: translateY(-2px);
    }

    .form-input::placeholder {
        color: var(--gray);
        opacity: 0.6;
    }

    /* Маска для ввода */
    .masked-input {
        font-family: monospace;
    }

    /* Иконки в полях */
    .input-with-icon {
        position: relative;
    }

    .input-icon {
        position: absolute;
        left: 20px;
        top: 50%;
        transform: translateY(-50%);
        font-size: 1.2rem;
        color: var(--gray);
        z-index: 1;
    }

    .input-with-icon .form-input {
        padding-left: 55px;
    }

    /* Группы полей */
    .form-row {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
    }

    @media (max-width: 576px) {
        .form-row {
            grid-template-columns: 1fr;
        }
    }

    /* Чекбокс согласия */
    .agree-section {
        background: rgba(76, 175, 125, 0.05);
        padding: 25px;
        border-radius: 15px;
        margin: 30px 0;
        border: 2px solid var(--light-green);
        transition: all 0.3s ease;
    }

    .agree-section:hover {
        border-color: var(--primary-green);
        transform: translateY(-3px);
    }

    .agree-checkbox {
        display: none;
    }

    .agree-label {
        display: flex;
        align-items: center;
        gap: 15px;
        cursor: pointer;
        color: var(--gray-dark);
        font-size: 1rem;
        line-height: 1.6;
    }

    .custom-checkbox {
        width: 24px;
        height: 24px;
        min-width: 24px;
        border: 2px solid var(--light-green);
        border-radius: 6px;
        background: var(--white);
        position: relative;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .custom-checkbox::after {
        content: '✓';
        color: var(--white);
        font-size: 14px;
        font-weight: bold;
        opacity: 0;
        transform: scale(0);
        transition: all 0.3s ease;
    }

    .agree-checkbox:checked + .agree-label .custom-checkbox {
        background: var(--primary-green);
        border-color: var(--primary-green);
    }

    .agree-checkbox:checked + .agree-label .custom-checkbox::after {
        opacity: 1;
        transform: scale(1);
    }

    .agree-link {
        color: var(--primary-green);
        text-decoration: none;
        font-weight: 600;
        transition: all 0.3s ease;
        border-bottom: 1px dotted transparent;
    }

    .agree-link:hover {
        color: var(--dark-green);
        border-bottom-color: var(--dark-green);
    }

    /* Кнопка отправки */
    .submit-section {
        margin-top: 40px;
    }

    .submit-button {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 15px;
        width: 100%;
        padding: 22px;
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: var(--white);
        border: none;
        border-radius: 15px;
        font-size: 1.2rem;
        font-weight: 700;
        cursor: pointer;
        transition: all 0.4s ease;
        position: relative;
        overflow: hidden;
    }

    .submit-button::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
        transition: left 0.5s ease;
    }

    .submit-button:hover::before {
        left: 100%;
    }

    .submit-button:hover {
        transform: translateY(-5px);
        box-shadow: 0 20px 40px rgba(76, 175, 125, 0.3);
    }

    .submit-button:active {
        transform: translateY(-2px);
    }

    .submit-button::after {
        content: '→';
        font-size: 1.5rem;
        transition: transform 0.3s ease;
    }

    .submit-button:hover::after {
        transform: translateX(10px);
    }

    /* Ссылки */
    .register-links {
        margin-top: 30px;
        text-align: center;
        color: var(--gray);
        padding-top: 25px;
        border-top: 1px solid var(--light-green);
    }

    .register-link {
        color: var(--primary-green);
        text-decoration: none;
        font-weight: 500;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 5px;
    }

    .register-link:hover {
        color: var(--dark-green);
        transform: translateY(-2px);
    }

    .register-link::after {
        content: '→';
        transition: transform 0.3s ease;
    }

    .register-link:hover::after {
        transform: translateX(5px);
    }

    /* Информационные сообщения */
    .register-info {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 20px;
        margin-top: 40px;
    }

    .info-card {
        background: var(--white);
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 10px 25px rgba(76, 175, 125, 0.1);
        text-align: center;
        transition: all 0.3s ease;
    }

    .info-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 35px rgba(76, 175, 125, 0.15);
    }

    .info-icon {
        font-size: 2.5rem;
        margin-bottom: 15px;
        display: block;
    }

    .info-card h4 {
        color: var(--dark-green);
        margin-bottom: 10px;
    }

    .info-card p {
        color: var(--gray);
        font-size: 0.9rem;
        line-height: 1.6;
    }

    /* Анимации */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(40px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    /* Валидация */
    .form-input.error {
        border-color: var(--error);
        animation: shake 0.5s ease;
    }

    @keyframes shake {
        0%, 100% { transform: translateX(0); }
        25% { transform: translateX(-5px); }
        75% { transform: translateX(5px); }
    }

    .error-message {
        color: var(--error);
        font-size: 0.9rem;
        margin-top: 5px;
        display: flex;
        align-items: center;
        gap: 5px;
    }

    .error-message::before {
        content: '⚠️';
    }

    /* Загрузка */
    .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(255,255,255,0.9);
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s ease;
    }

    .loading-overlay.active {
        opacity: 1;
        visibility: visible;
    }

    .loading-spinner {
        width: 50px;
        height: 50px;
        border: 4px solid var(--light-green);
        border-top-color: var(--primary-green);
        border-radius: 50%;
        animation: spin 1s linear infinite;
        margin-bottom: 20px;
    }

    @keyframes spin {
        to { transform: rotate(360deg); }
    }

    /* Кнопка назад */
    .back-button {
        display: inline-flex;
        align-items: center;
        gap: 10px;
        padding: 12px 25px;
        background: var(--white);
        color: var(--gray-dark);
        text-decoration: none;
        border-radius: 50px;
        font-weight: 500;
        margin-bottom: 30px;
        transition: all 0.3s ease;
        box-shadow: 0 5px 15px rgba(76, 175, 125, 0.1);
    }

    .back-button:hover {
        background: var(--light-green);
        color: var(--dark-green);
        transform: translateX(-5px);
    }

    .back-button::before {
        content: '←';
        transition: transform 0.3s ease;
    }

    .back-button:hover::before {
        transform: translateX(-5px);
    }

    /* Индикатор сложности пароля */
    .password-strength {
        margin-top: 10px;
        display: none;
    }

    .strength-bar {
        height: 6px;
        background: var(--light-green);
        border-radius: 3px;
        overflow: hidden;
        margin-top: 5px;
    }

    .strength-fill {
        height: 100%;
        width: 0%;
        background: var(--error);
        transition: all 0.3s ease;
    }

    .strength-text {
        font-size: 0.85rem;
        color: var(--gray);
        margin-top: 5px;
    }

    /* Адаптивность */
    @media (max-width: 576px) {
        .register-title {
            font-size: 2.2rem;
        }

        .form-card {
            padding: 25px;
        }

        .register-info {
            grid-template-columns: 1fr;
        }

        .agree-label {
            font-size: 0.9rem;
        }
    }

    /* Кастомный скроллбар */
    ::-webkit-scrollbar {
        width: 8px;
    }

    ::-webkit-scrollbar-track {
        background: var(--light-green);
        border-radius: 4px;
    }

    ::-webkit-scrollbar-thumb {
        background: var(--primary-green);
        border-radius: 4px;
    }

    ::-webkit-scrollbar-thumb:hover {
        background: var(--dark-green);
    }

    /* Хинты для полей */
    .field-hint {
        font-size: 0.85rem;
        color: var(--gray);
        margin-top: 5px;
        display: flex;
        align-items: center;
        gap: 5px;
    }

    .field-hint::before {
        content: '💡';
    }
</style>

<div class="register-page">
    <!-- Анимированный фон -->
    <div class="register-background">
        <div class="register-shape"></div>
        <div class="register-shape"></div>
        <div class="register-shape"></div>
    </div>

    <div class="container register-container">
        <!-- Кнопка назад -->
        <a href="/" class="back-button reveal">
            Вернуться на главную
        </a>

        <!-- Заголовок -->
        <div class="register-header reveal">
            <h1 class="register-title">Регистрация</h1>
            <p class="register-subtitle">Создайте аккаунт для бронирования туров и получения персональных предложений</p>
        </div>

        <!-- Основной контент -->
        <div class="register-form-section reveal">
            <div class="form-card">
                <?php $form = ActiveForm::begin([
                    'id' => 'register-form',
                    'enableClientValidation' => true,
                ]); ?>

                <!-- ФИО -->
                <div class="form-group">
                    <label class="form-label">
                        <span>👤</span>
                        ФИО
                    </label>
                    <div class="input-with-icon">
                        <span class="input-icon">📝</span>
                        <?= $form->field($model, 'full_name', [
                            'template' => '{input}{error}',
                        ])->textInput([
                            'class' => 'form-input',
                            'required' => true,
                            'placeholder' => 'Иванов Иван Иванович'
                        ])->label(false) ?>
                    </div>
                </div>

                <!-- Контакты в ряд -->
                <div class="form-row">
                    <!-- Телефон -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>📱</span>
                            Телефон
                        </label>
                        <div class="input-with-icon">
                            <span class="input-icon">📞</span>
                            <?= $form->field($model, 'phone', [
                                'template' => '{input}{error}',
                            ])->widget(MaskedInput::class, [
                                'mask' => '+7(999)999-99-99',
                                'options' => [
                                    'class' => 'form-input masked-input',
                                    'required' => true,
                                    'placeholder' => '+7(___)___-__-__'
                                ]
                            ])->label(false) ?>
                        </div>
                    </div>

                    <!-- Email -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>📧</span>
                            Email
                        </label>
                        <div class="input-with-icon">
                            <span class="input-icon">✉️</span>
                            <?= $form->field($model, 'email', [
                                'template' => '{input}{error}',
                            ])->input('email', [
                                'class' => 'form-input',
                                'required' => true,
                                'placeholder' => 'example@mail.ru'
                            ])->label(false) ?>
                        </div>
                    </div>
                </div>

                <!-- Паспорт и адрес -->
                <div class="form-row">
                    <!-- Паспортные данные -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>🆔</span>
                            Паспорт
                        </label>
                        <div class="input-with-icon">
                            <span class="input-icon">🔖</span>
                            <?= $form->field($model, 'passport', [
                                'template' => '{input}{error}',
                            ])->widget(MaskedInput::class, [
                                'mask' => '9999 999999',
                                'options' => [
                                    'class' => 'form-input masked-input',
                                    'required' => true,
                                    'placeholder' => '1234 567890'
                                ]
                            ])->label(false) ?>
                        </div>
                        <div class="field-hint">Серия и номер без пробелов</div>
                    </div>

                    <!-- Адрес -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>🏠</span>
                            Адрес
                        </label>
                        <div class="input-with-icon">
                            <span class="input-icon">📍</span>
                            <?= $form->field($model, 'address', [
                                'template' => '{input}{error}',
                            ])->textInput([
                                'class' => 'form-input',
                                'required' => true,
                                'placeholder' => 'г. Москва, ул. Примерная, д. 1'
                            ])->label(false) ?>
                        </div>
                    </div>
                </div>

                <!-- Пароли в ряд -->
                <div class="form-row">
                    <!-- Пароль -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>🔒</span>
                            Пароль
                        </label>
                        <div class="input-with-icon">
                            <span class="input-icon">🔑</span>
                            <?= $form->field($model, 'password', [
                                'template' => '{input}{error}',
                            ])->passwordInput([
                                'class' => 'form-input',
                                'required' => true,
                                'placeholder' => 'Придумайте пароль',
                                'id' => 'password-input'
                            ])->label(false) ?>
                        </div>
                        <div class="password-strength" id="password-strength">
                            <div class="strength-bar">
                                <div class="strength-fill" id="strength-fill"></div>
                            </div>
                            <div class="strength-text" id="strength-text">Слабый пароль</div>
                        </div>
                    </div>

                    <!-- Повтор пароля -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>🔁</span>
                            Повторите пароль
                        </label>
                        <div class="input-with-icon">
                            <span class="input-icon">🔐</span>
                            <?= $form->field($model, 'password_repeat', [
                                'template' => '{input}{error}',
                            ])->passwordInput([
                                'class' => 'form-input',
                                'required' => true,
                                'placeholder' => 'Повторите пароль',
                                'id' => 'password-repeat'
                            ])->label(false) ?>
                        </div>
                        <div id="password-match" style="display: none;">
                            <div class="field-hint" style="color: var(--primary-green);">✓ Пароли совпадают</div>
                        </div>
                    </div>
                </div>

                <!-- Согласие с правилами -->
                <div class="agree-section">
                    <input type="checkbox"
                           id="agree-checkbox"
                           name="RegisterForm[agree]"
                           value="1"
                           class="agree-checkbox"
                           required>
                    <label for="agree-checkbox" class="agree-label">
                        <div class="custom-checkbox"></div>
                        <div>
                            Согласен с
                            <a href="#" class="agree-link">правилами регистрации</a>
                            и обработки персональных данных
                        </div>
                    </label>
                    <?php if ($model->hasErrors('agree')): ?>
                        <div class="error-message">
                            <?= Html::encode($model->getFirstError('agree')) ?>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Кнопка отправки -->
                <div class="submit-section">
                    <button type="submit" class="submit-button">
                        <span>Создать аккаунт</span>
                    </button>
                </div>

                <?php ActiveForm::end(); ?>

                <!-- Ссылка на вход -->
                <div class="register-links">
                    <p><a href="/site/login" class="register-link">Уже есть аккаунт? Войти</a></p>
                </div>
            </div>

        </div>

        <!-- Оверлей загрузки -->
        <div class="loading-overlay" id="loadingOverlay">
            <div class="loading-spinner"></div>
            <p style="color: var(--dark-green); font-weight: 600;">Регистрируем аккаунт...</p>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Инициализация анимаций
        const reveals = document.querySelectorAll('.reveal');

        function reveal() {
            reveals.forEach(element => {
                const windowHeight = window.innerHeight;
                const elementTop = element.getBoundingClientRect().top;
                const elementVisible = 100;

                if (elementTop < windowHeight - elementVisible) {
                    element.classList.add('active');
                }
            });
        }

        reveal();
        window.addEventListener('scroll', reveal);

        // Элементы формы
        const registerForm = document.getElementById('register-form');
        const loadingOverlay = document.getElementById('loadingOverlay');
        const passwordInput = document.getElementById('password-input');
        const passwordRepeat = document.getElementById('password-repeat');
        const passwordMatch = document.getElementById('password-match');
        const strengthIndicator = document.getElementById('password-strength');
        const strengthFill = document.getElementById('strength-fill');
        const strengthText = document.getElementById('strength-text');
        const agreeCheckbox = document.getElementById('agree-checkbox');
        const agreeSection = document.querySelector('.agree-section');

        // Проверка сложности пароля
        function checkPasswordStrength(password) {
            let score = 0;

            if (password.length >= 8) score++;
            if (/[A-Z]/.test(password)) score++;
            if (/[a-z]/.test(password)) score++;
            if (/[0-9]/.test(password)) score++;
            if (/[^A-Za-z0-9]/.test(password)) score++;

            let strength = 'очень слабый';
            let color = var(--error);
            let width = '20%';

            if (score >= 4) {
                strength = 'сильный';
                color = var(--primary-green);
                width = '100%';
            } else if (score >= 3) {
                strength = 'средний';
                color = var(--warning);
                width = '70%';
            } else if (score >= 2) {
                strength = 'слабый';
                color = var(--warning);
                width = '40%';
            }

            return { strength, color, width };
        }

        // Обновление индикатора сложности пароля
        passwordInput.addEventListener('input', function() {
            const password = this.value;

            if (password.length > 0) {
                strengthIndicator.style.display = 'block';
                const { strength, color, width } = checkPasswordStrength(password);

                strengthFill.style.width = width;
                strengthFill.style.background = color;
                strengthText.textContent = `Пароль: ${strength}`;
                strengthText.style.color = color;
            } else {
                strengthIndicator.style.display = 'none';
            }

            // Проверка совпадения паролей
            checkPasswordMatch();
        });

        // Проверка совпадения паролей
        passwordRepeat.addEventListener('input', checkPasswordMatch);

        function checkPasswordMatch() {
            const password = passwordInput.value;
            const repeat = passwordRepeat.value;

            if (repeat.length > 0) {
                if (password === repeat && password.length > 0) {
                    passwordMatch.style.display = 'block';
                    passwordRepeat.classList.remove('error');
                    passwordRepeat.classList.add('success');
                } else {
                    passwordMatch.style.display = 'none';
                    if (password.length > 0 && repeat.length > 0) {
                        passwordRepeat.classList.add('error');
                        passwordRepeat.classList.remove('success');
                    }
                }
            } else {
                passwordMatch.style.display = 'none';
                passwordRepeat.classList.remove('error', 'success');
            }
        }

        // Валидация формы
        registerForm.addEventListener('submit', function(e) {
            let isValid = true;

            // Проверка обязательных полей
            const requiredFields = registerForm.querySelectorAll('[required]');

            requiredFields.forEach(field => {
                if (field.type === 'checkbox') {
                    if (!field.checked) {
                        field.classList.add('error');
                        isValid = false;

                        // Анимация для чекбокса
                        agreeSection.style.animation = 'none';
                        setTimeout(() => {
                            agreeSection.style.animation = 'shake 0.5s ease';
                        }, 10);
                    } else {
                        field.classList.remove('error');
                    }
                } else if (!field.value.trim()) {
                    field.classList.add('error');
                    isValid = false;

                    // Анимация ошибки
                    field.style.animation = 'none';
                    setTimeout(() => {
                        field.style.animation = 'shake 0.5s ease';
                    }, 10);
                } else {
                    field.classList.remove('error');
                }
            });

            // Проверка совпадения паролей
            if (passwordInput.value !== passwordRepeat.value) {
                passwordInput.classList.add('error');
                passwordRepeat.classList.add('error');
                isValid = false;

                // Анимация для паролей
                passwordInput.style.animation = 'none';
                passwordRepeat.style.animation = 'none';
                setTimeout(() => {
                    passwordInput.style.animation = 'shake 0.5s ease';
                    passwordRepeat.style.animation = 'shake 0.5s ease';
                }, 10);
            }

            if (isValid) {
                loadingOverlay.classList.add('active');

                // Симуляция загрузки (в реальном приложении удалить)
                setTimeout(() => {
                    loadingOverlay.classList.remove('active');
                }, 2000);
            } else {
                e.preventDefault();

                // Прокрутка к первой ошибке
                const firstError = registerForm.querySelector('.error');
                if (firstError) {
                    firstError.scrollIntoView({
                        behavior: 'smooth',
                        block: 'center'
                    });
                }
            }
        });

        // Удаление класса ошибки при вводе
        registerForm.querySelectorAll('input').forEach(field => {
            field.addEventListener('input', function() {
                this.classList.remove('error');

                // Убираем анимацию у родительского элемента для чекбокса
                if (this.type === 'checkbox' && agreeSection) {
                    agreeSection.style.animation = '';
                }
            });
        });

        // Анимация при фокусе
        registerForm.querySelectorAll('.form-input').forEach(field => {
            field.addEventListener('focus', function() {
                const parent = this.closest('.form-group');
                if (parent) {
                    parent.style.transform = 'translateY(-5px)';
                }
            });

            field.addEventListener('blur', function() {
                const parent = this.closest('.form-group');
                if (parent) {
                    parent.style.transform = 'translateY(0)';
                }
            });
        });

        // Параллакс эффект для фона
        window.addEventListener('scroll', function() {
            const scrolled = window.pageYOffset;
            const background = document.querySelector('.register-background');
            if (background) {
                background.style.transform = `translateY(${scrolled * 0.05}px)`;
            }
        });

        // Анимация для чекбокса
        if (agreeCheckbox) {
            agreeCheckbox.addEventListener('change', function() {
                if (this.checked) {
                    agreeSection.style.borderColor = var(--primary-green);
                    agreeSection.style.transform = 'translateY(-3px)';

                    // Анимация галочки
                    const checkbox = this.nextElementSibling.querySelector('.custom-checkbox');
                    checkbox.style.transform = 'scale(1.1)';
                    setTimeout(() => {
                        checkbox.style.transform = 'scale(1)';
                    }, 300);
                } else {
                    agreeSection.style.borderColor = var(--light-green);
                    agreeSection.style.transform = 'translateY(0)';
                }
            });
        }

        // Маска для телефона - улучшенная обработка
        const phoneInput = registerForm.querySelector('input[type="tel"]');
        if (phoneInput) {
            phoneInput.addEventListener('input', function() {
                const value = this.value.replace(/\D/g, '');
                if (value.length === 11 && value.startsWith('7')) {
                    this.classList.remove('error');
                }
            });
        }

        // Маска для паспорта
        const passportInput = registerForm.querySelector('input[placeholder="1234 567890"]');
        if (passportInput) {
            passportInput.addEventListener('input', function() {
                const value = this.value.replace(/\D/g, '');
                if (value.length === 10) {
                    this.classList.remove('error');
                }
            });
        }

        // Эффект при наведении на ссылки
        document.querySelectorAll('.register-link, .agree-link').forEach(link => {
            link.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-2px)';
            });

            link.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
            });
        });

        // Инициализация CSS переменных для JS
        const root = document.documentElement;
        const computedStyle = getComputedStyle(root);

        // Функция для получения CSS переменных
        function getCssVariable(name) {
            return computedStyle.getPropertyValue('--' + name).trim();
        }

        // Используем CSS переменные в JS
        window.var = function(name) {
            return getCssVariable(name);
        };

        // Автозаполнение даты рождения (опционально)
        const today = new Date();
        const minDate = new Date(today.getFullYear() - 100, today.getMonth(), today.getDate());
        const maxDate = new Date(today.getFullYear() - 18, today.getMonth(), today.getDate());

        // Если нужно добавить поле даты рождения
        // dateInput.min = minDate.toISOString().split('T')[0];
        // dateInput.max = maxDate.toISOString().split('T')[0];
    });
</script>