<?php
use yii\helpers\Html;
use yii\helpers\Url;
use yii\widgets\ActiveForm;

$this->title = 'Каталог туров | Weekend Tours';
?>

<style>
    :root {
        --primary-green: #4CAF7D;
        --light-green: #E8F5E9;
        --accent-green: #81C784;
        --dark-green: #388E3C;
        --white: #FFFFFF;
        --gray-light: #F5F5F5;
        --gray: #757575;
        --gray-dark: #424242;
    }

    .catalog-page {
        min-height: 100vh;
        background: linear-gradient(135deg, var(--light-green) 0%, var(--white) 100%);
        padding: 40px 0;
    }

    /* Анимированный заголовок */
    .catalog-header {
        text-align: center;
        margin-bottom: 60px;
        position: relative;
    }

    .catalog-title {
        font-size: 3.5rem;
        font-weight: 800;
        background: linear-gradient(45deg, var(--dark-green), var(--primary-green));
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        margin-bottom: 20px;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease forwards;
    }

    .catalog-subtitle {
        font-size: 1.2rem;
        color: var(--gray);
        max-width: 600px;
        margin: 0 auto;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.3s forwards;
    }

    /* Фильтры */
    .filters-section {
        background: var(--white);
        padding: 30px;
        border-radius: 20px;
        box-shadow: 0 10px 30px rgba(76, 175, 125, 0.1);
        margin-bottom: 50px;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.5s forwards;
    }

    .filter-form {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 25px;
        align-items: end;
    }

    .filter-group {
        position: relative;
    }

    .filter-label {
        display: block;
        margin-bottom: 10px;
        color: var(--dark-green);
        font-weight: 600;
        font-size: 0.9rem;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    .filter-input {
        width: 100%;
        padding: 15px 20px;
        border: 2px solid var(--light-green);
        border-radius: 12px;
        font-size: 1rem;
        color: var(--gray-dark);
        background: var(--white);
        transition: all 0.3s ease;
        outline: none;
    }

    .filter-input:focus {
        border-color: var(--primary-green);
        box-shadow: 0 0 0 3px rgba(76, 175, 125, 0.1);
        transform: translateY(-2px);
    }

    .filter-input::placeholder {
        color: var(--gray);
        opacity: 0.7;
    }

    .filter-button {
        padding: 15px 40px;
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: var(--white);
        border: none;
        border-radius: 12px;
        font-size: 1rem;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        height: 52px;
        position: relative;
        overflow: hidden;
    }

    .filter-button:hover {
        transform: translateY(-3px);
        box-shadow: 0 15px 25px rgba(76, 175, 125, 0.2);
    }

    .filter-button::after {
        position: absolute;
        right: 20px;
        top: 50%;
        transform: translateY(-50%);
        opacity: 0;
        transition: all 0.3s ease;
    }

    .filter-button:hover::after {
        opacity: 1;
        right: 25px;
    }

    .reset-button {
        padding: 15px 20px;
        background: var(--gray-light);
        color: var(--gray-dark);
        border: none;
        border-radius: 12px;
        font-size: 1rem;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.3s ease;
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
    }

    .reset-button:hover {
        background: #e0e0e0;
        transform: translateY(-2px);
        color: var(--dark-green);
    }

    /* Сетка туров */
    .tours-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
        gap: 40px;
        margin-top: 30px;
    }

    .tour-card {
        background: var(--white);
        border-radius: 20px;
        overflow: hidden;
        box-shadow: 0 10px 30px rgba(76, 175, 125, 0.1);
        transition: all 0.4s ease;
        opacity: 0;
        transform: translateY(40px);
        animation: fadeInUp 0.8s ease forwards;
    }

    .tour-card:nth-child(1) { animation-delay: 0.1s; }
    .tour-card:nth-child(2) { animation-delay: 0.2s; }
    .tour-card:nth-child(3) { animation-delay: 0.3s; }
    .tour-card:nth-child(4) { animation-delay: 0.4s; }
    .tour-card:nth-child(5) { animation-delay: 0.5s; }
    .tour-card:nth-child(6) { animation-delay: 0.6s; }

    .tour-card:hover {
        transform: translateY(-10px) scale(1.02);
        box-shadow: 0 25px 50px rgba(76, 175, 125, 0.15);
    }

    .tour-image-container {
        position: relative;
        overflow: hidden;
        height: 250px;
    }

    .tour-image {
        width: 100%;
        height: 100%;
        object-fit: cover;
        transition: transform 0.6s ease;
    }

    .tour-card:hover .tour-image {
        transform: scale(1.1);
    }

    .tour-badge {
        position: absolute;
        top: 20px;
        right: 20px;
        background: var(--primary-green);
        color: var(--white);
        padding: 8px 16px;
        border-radius: 20px;
        font-size: 0.9rem;
        font-weight: 600;
        transform: translateX(20px);
        opacity: 0;
        transition: all 0.3s ease 0.2s;
    }

    .tour-card:hover .tour-badge {
        transform: translateX(0);
        opacity: 1;
    }

    .tour-content {
        padding: 25px;
    }

    .tour-title {
        font-size: 1.4rem;
        color: var(--dark-green);
        margin-bottom: 15px;
        line-height: 1.4;
        transition: color 0.3s ease;
    }

    .tour-card:hover .tour-title {
        color: var(--primary-green);
    }

    .tour-description {
        color: var(--gray);
        margin-bottom: 20px;
        line-height: 1.6;
        font-size: 0.95rem;
        display: -webkit-box;
        -webkit-line-clamp: 3;
        -webkit-box-orient: vertical;
        overflow: hidden;
    }

    .tour-meta {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 25px;
        padding-top: 15px;
        border-top: 2px solid var(--light-green);
    }

    .tour-price {
        font-size: 1.8rem;
        color: var(--primary-green);
        font-weight: 700;
    }

    .tour-price::after {
        content: '₽';
        font-size: 1.2rem;
        margin-left: 4px;
    }

    .tour-duration {
        display: flex;
        align-items: center;
        gap: 8px;
        color: var(--gray);
        font-size: 0.9rem;
    }

    .tour-button {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        gap: 10px;
        padding: 14px 30px;
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: var(--white);
        text-decoration: none;
        border-radius: 12px;
        font-weight: 600;
        transition: all 0.3s ease;
        width: 100%;
        border: none;
        cursor: pointer;
    }

    .tour-button:hover {
        transform: translateY(-3px);
        box-shadow: 0 10px 20px rgba(76, 175, 125, 0.3);
        color: var(--white);
    }

    .tour-button::after {
        content: '→';
        font-size: 1.2rem;
        transition: transform 0.3s ease;
    }

    .tour-button:hover::after {
        transform: translateX(5px);
    }

    /* Пустой результат */
    .no-results {
        text-align: center;
        padding: 80px 20px;
        grid-column: 1 / -1;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.7s forwards;
    }

    .no-results-icon {
        font-size: 4rem;
        margin-bottom: 20px;
        color: var(--primary-green);
    }

    .no-results h3 {
        color: var(--dark-green);
        font-size: 1.8rem;
        margin-bottom: 15px;
    }

    .no-results p {
        color: var(--gray);
        font-size: 1.1rem;
        margin-bottom: 30px;
        max-width: 500px;
        margin-left: auto;
        margin-right: auto;
    }

    .reset-all-button {
        display: inline-flex;
        align-items: center;
        gap: 10px;
        padding: 15px 40px;
        background: var(--white);
        color: var(--dark-green);
        text-decoration: none;
        border-radius: 12px;
        font-weight: 600;
        border: 2px solid var(--light-green);
        transition: all 0.3s ease;
    }

    .reset-all-button:hover {
        background: var(--primary-green);
        color: var(--white);
        border-color: var(--primary-green);
        transform: translateY(-3px);
        box-shadow: 0 10px 20px rgba(76, 175, 125, 0.2);
    }

    /* Пагинация */
    .pagination-container {
        margin-top: 60px;
        text-align: center;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.8s forwards;
    }

    .pagination {
        display: inline-flex;
        gap: 10px;
        background: var(--white);
        padding: 15px;
        border-radius: 12px;
        box-shadow: 0 5px 20px rgba(76, 175, 125, 0.1);
    }

    .page-item {
        list-style: none;
    }

    .page-link {
        display: flex;
        align-items: center;
        justify-content: center;
        width: 45px;
        height: 45px;
        border-radius: 10px;
        color: var(--gray-dark);
        text-decoration: none;
        font-weight: 600;
        transition: all 0.3s ease;
    }

    .page-link:hover {
        background: var(--light-green);
        color: var(--dark-green);
        transform: translateY(-2px);
    }

    .page-link.active {
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: var(--white);
        box-shadow: 0 5px 15px rgba(76, 175, 125, 0.2);
    }

    /* Анимации */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(40px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    /* Адаптивность */
    @media (max-width: 768px) {
        .catalog-title {
            font-size: 2.5rem;
        }

        .filter-form {
            grid-template-columns: 1fr;
        }

        .tours-grid {
            grid-template-columns: 1fr;
        }

        .tour-button-container {
            flex-direction: column;
            gap: 10px;
        }
    }

    /* Дополнительная информация о турах */
    .tour-extra {
        display: flex;
        gap: 15px;
        margin-top: 15px;
        flex-wrap: wrap;
    }

    .tour-tag {
        padding: 6px 12px;
        background: var(--light-green);
        color: var(--dark-green);
        border-radius: 20px;
        font-size: 0.8rem;
        font-weight: 500;
    }

    /* Анимация загрузки */
    @keyframes shimmer {
        0% {
            background-position: -1000px 0;
        }
        100% {
            background-position: 1000px 0;
        }
    }

    .loading-skeleton {
        background: linear-gradient(90deg, var(--gray-light) 25%, #e8e8e8 50%, var(--gray-light) 75%);
        background-size: 1000px 100%;
        animation: shimmer 2s infinite;
        border-radius: 10px;
    }
</style>

<div class="catalog-page">
    <div class="container">
        <!-- Заголовок -->
        <div class="catalog-header reveal">
            <h1 class="catalog-title">Каталог туров</h1>
            <p class="catalog-subtitle">
                Найдите идеальное путешествие выходного дня среди лучших маршрутов Урала
            </p>
        </div>

        <!-- Форма фильтрации -->
        <div class="filters-section reveal">
            <?php $form = ActiveForm::begin([
                'method' => 'get',
                'action' => Url::to(['/site/catalog']),
                'options' => ['class' => 'filter-form']
            ]); ?>

            <div class="filter-group">
                <label class="filter-label">Минимальная цена</label>
                <?= $form->field($filterModel, 'minPrice', [
                    'template' => '{input}',
                    'options' => ['class' => '']
                ])->textInput([
                    'type' => 'number',
                    'min' => 0,
                    'placeholder' => 'от 0 ₽',
                    'class' => 'filter-input'
                ])->label(false) ?>
            </div>

            <div class="filter-group">
                <label class="filter-label">Максимальная цена</label>
                <?= $form->field($filterModel, 'maxPrice', [
                    'template' => '{input}',
                    'options' => ['class' => '']
                ])->textInput([
                    'type' => 'number',
                    'min' => 0,
                    'placeholder' => 'до 100000 ₽',
                    'class' => 'filter-input'
                ])->label(false) ?>
            </div>

            <div class="filter-group">
                <button type="submit" class="filter-button">
                    Применить фильтры
                </button>
            </div>

            <div class="filter-group">
                <a href="<?= Url::to(['/site/catalog']) ?>" class="reset-button">
                    <span>↻</span>
                    Сбросить
                </a>
            </div>

            <?php ActiveForm::end(); ?>
        </div>

        <!-- Сетка туров -->
        <div class="tours-grid">
            <?php if (!empty($tours)): ?>
                <?php foreach ($tours as $index => $tour): ?>
                    <div class="tour-card reveal">
                        <!-- Изображение тура -->
                        <?php if ($tour->image): ?>
                            <div class="tour-image-container">
                                <img src="<?= Url::to(['/images/' . $tour->image]) ?>"
                                     class="tour-image"
                                     alt="<?= Html::encode($tour->title) ?>">
                                <div class="tour-badge">Популярный</div>
                            </div>
                        <?php endif; ?>

                        <!-- Контент тура -->
                        <div class="tour-content">
                            <h3 class="tour-title"><?= Html::encode($tour->title) ?></h3>
                            <p class="tour-description">
                                <?= Html::encode(mb_substr($tour->description, 0, 150)) ?>...
                            </p>

                            <!-- Дополнительные теги -->
                            <div class="tour-extra">
                                <span class="tour-tag">Групповой</span>
                                <span class="tour-tag">Треккинг</span>
                            </div>

                            <div class="tour-meta">
                                <div class="tour-price"><?= Html::encode($tour->price) ?></div>
                                <div class="tour-duration">
                                </div>
                            </div>

                            <?= Html::a('Подробнее о туре',
                                ['/site/tour', 'id' => $tour->id],
                                ['class' => 'tour-button']
                            ) ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <!-- Сообщение при отсутствии результатов -->
                <div class="no-results reveal">
                    <div class="no-results-icon">🔍</div>
                    <h3>Туры не найдены</h3>
                    <p>К сожалению, по вашим критериям фильтрации туров не найдено.</p>
                    <p>Попробуйте изменить параметры поиска или сбросить фильтры.</p>
                    <a href="<?= Url::to(['/site/catalog']) ?>" class="reset-all-button">
                        <span>↻</span>
                        Сбросить все фильтры
                    </a>
                </div>
            <?php endif; ?>
        </div>

        <!-- Пагинация (если есть) -->
        <?php if (!empty($pagination)): ?>
            <div class="pagination-container reveal">
                <nav aria-label="Навигация по страницам">
                    <ul class="pagination">
                        <?php foreach ($pagination as $page): ?>
                            <li class="page-item">
                                <?= Html::a($page['label'], $page['url'], [
                                    'class' => 'page-link' . ($page['active'] ? ' active' : '')
                                ]) ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </nav>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
    // Анимация появления при скролле
    document.addEventListener('DOMContentLoaded', function() {
        const reveals = document.querySelectorAll('.reveal');

        function reveal() {
            reveals.forEach(element => {
                const windowHeight = window.innerHeight;
                const elementTop = element.getBoundingClientRect().top;
                const elementVisible = 100;

                if (elementTop < windowHeight - elementVisible) {
                    element.classList.add('active');
                }
            });
        }

        // Запуск анимаций
        reveal();
        window.addEventListener('scroll', reveal);

        // Плавная прокрутка к фильтрам при их использовании
        const filterInputs = document.querySelectorAll('.filter-input');
        filterInputs.forEach(input => {
            input.addEventListener('focus', function() {
                this.parentElement.style.transform = 'scale(1.02)';
            });

            input.addEventListener('blur', function() {
                this.parentElement.style.transform = 'scale(1)';
            });
        });

        // Анимация кнопки сброса
        const resetButton = document.querySelector('.reset-button');
        if (resetButton) {
            resetButton.addEventListener('mouseenter', function() {
                this.querySelector('span').style.transform = 'rotate(180deg)';
            });

            resetButton.addEventListener('mouseleave', function() {
                this.querySelector('span').style.transform = 'rotate(0deg)';
            });
        }

        // Подсветка активных фильтров
        const urlParams = new URLSearchParams(window.location.search);
        const minPrice = urlParams.get('minPrice');
        const maxPrice = urlParams.get('maxPrice');

        if (minPrice || maxPrice) {
            const filterSection = document.querySelector('.filters-section');
            if (filterSection) {
                filterSection.style.boxShadow = '0 10px 30px rgba(76, 175, 125, 0.2)';
                filterSection.style.borderLeft = '4px solid var(--primary-green)';
            }
        }
    });

    // Анимация для карточек при загрузке
    window.addEventListener('load', function() {
        const cards = document.querySelectorAll('.tour-card');
        cards.forEach((card, index) => {
            card.style.animationDelay = (index * 0.1) + 's';
        });
    });
</script>