<?php
use yii\helpers\Html;
use yii\widgets\ActiveForm;
use yii\widgets\MaskedInput;

$this->title = 'Оформление заявки | ' . Html::encode($tour->title);
?>

<style>
    :root {
        --primary-green: #4CAF7D;
        --light-green: #E8F5E9;
        --accent-green: #81C784;
        --dark-green: #388E3C;
        --white: #FFFFFF;
        --gray-light: #F5F5F5;
        --gray: #757575;
        --gray-dark: #424242;
        --warning: #FF9800;
        --error: #f44336;
    }

    .booking-page {
        min-height: 100vh;
        background: linear-gradient(135deg, var(--light-green) 0%, var(--white) 100%);
        padding: 40px 0;
        position: relative;
        overflow: hidden;
    }

    /* Анимированный фон */
    .booking-background {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        z-index: 0;
        opacity: 0.1;
    }

    .booking-shape {
        position: absolute;
        border-radius: 50%;
        background: var(--primary-green);
        filter: blur(60px);
        opacity: 0.3;
        animation: float 20s infinite ease-in-out;
    }

    .booking-shape:nth-child(1) {
        width: 400px;
        height: 400px;
        top: -100px;
        right: -100px;
        animation-delay: 0s;
    }

    .booking-shape:nth-child(2) {
        width: 300px;
        height: 300px;
        bottom: -50px;
        left: -50px;
        animation-delay: -5s;
    }

    @keyframes float {
        0%, 100% {
            transform: translate(0, 0) rotate(0deg);
        }
        33% {
            transform: translate(30px, 30px) rotate(120deg);
        }
        66% {
            transform: translate(-20px, 40px) rotate(240deg);
        }
    }

    /* Основной контейнер */
    .booking-container {
        position: relative;
        z-index: 1;
        max-width: 900px;
        margin: 0 auto;
    }

    /* Заголовок */
    .booking-header {
        text-align: center;
        margin-bottom: 50px;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease forwards;
    }

    .booking-title {
        font-size: 3rem;
        font-weight: 800;
        background: linear-gradient(45deg, var(--dark-green), var(--primary-green));
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        margin-bottom: 15px;
    }

    .booking-subtitle {
        font-size: 1.2rem;
        color: var(--gray);
        max-width: 600px;
        margin: 0 auto;
    }

    .tour-info-card {
        background: var(--white);
        border-radius: 20px;
        padding: 25px;
        margin-top: 20px;
        box-shadow: 0 10px 30px rgba(76, 175, 125, 0.1);
        display: flex;
        align-items: center;
        gap: 20px;
        transition: all 0.3s ease;
    }

    .tour-info-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 40px rgba(76, 175, 125, 0.15);
    }

    .tour-image-small {
        width: 100px;
        height: 100px;
        border-radius: 15px;
        object-fit: cover;
    }

    .tour-details h3 {
        color: var(--dark-green);
        margin-bottom: 10px;
        font-size: 1.3rem;
    }

    .tour-details p {
        color: var(--gray);
        margin: 5px 0;
    }

    .tour-price-badge {
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: var(--white);
        padding: 8px 20px;
        border-radius: 50px;
        font-weight: 700;
        font-size: 1.2rem;
        display: inline-block;
    }

    /* Основной контент */
    .booking-content {
        display: grid;
        grid-template-columns: 1fr 350px;
        gap: 40px;
        margin-bottom: 60px;
    }

    @media (max-width: 768px) {
        .booking-content {
            grid-template-columns: 1fr;
        }
    }

    /* Форма */
    .booking-form-section {
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.3s forwards;
    }

    .form-card {
        background: var(--white);
        border-radius: 20px;
        padding: 40px;
        box-shadow: 0 15px 40px rgba(76, 175, 125, 0.1);
    }

    /* Поля формы */
    .form-group {
        margin-bottom: 30px;
        position: relative;
    }

    .form-label {
        display: block;
        margin-bottom: 12px;
        color: var(--dark-green);
        font-weight: 600;
        font-size: 1rem;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .form-label::after {
        content: '*';
        color: var(--error);
        font-size: 1.2rem;
        opacity: 0.7;
    }

    .form-input,
    .form-select,
    .form-textarea {
        width: 100%;
        padding: 18px 20px;
        border: 2px solid var(--light-green);
        border-radius: 12px;
        font-size: 1rem;
        color: var(--gray-dark);
        background: var(--white);
        transition: all 0.3s ease;
        outline: none;
        font-family: inherit;
    }

    .form-input:focus,
    .form-select:focus,
    .form-textarea:focus {
        border-color: var(--primary-green);
        box-shadow: 0 0 0 4px rgba(76, 175, 125, 0.1);
        transform: translateY(-2px);
    }

    .form-input::placeholder {
        color: var(--gray);
        opacity: 0.6;
    }

    .form-textarea {
        min-height: 120px;
        resize: vertical;
    }

    /* Кастомный календарь */
    .custom-date-input {
        position: relative;
        cursor: pointer;
    }

    .custom-date-input input {
        cursor: pointer;
    }

    .custom-date-input::after {
        content: '📅';
        position: absolute;
        right: 20px;
        top: 50%;
        transform: translateY(-50%);
        pointer-events: none;
        font-size: 1.2rem;
        z-index: 1;
    }

    /* Кастомный календарь - стилизация нативного datepicker */
    input[type="date"]::-webkit-calendar-picker-indicator {
        background: transparent;
        bottom: 0;
        color: transparent;
        cursor: pointer;
        height: auto;
        left: 0;
        position: absolute;
        right: 0;
        top: 0;
        width: auto;
        opacity: 0;
        z-index: 2;
    }

    input[type="date"]::-webkit-datetime-edit-fields-wrapper {
        color: var(--gray-dark);
    }

    input[type="date"]::-webkit-datetime-edit {
        color: var(--gray-dark);
    }

    input[type="date"]::-webkit-datetime-edit-text {
        color: var(--gray);
    }

    input[type="date"]::-webkit-datetime-edit-month-field {
        color: var(--gray);
    }

    input[type="date"]::-webkit-datetime-edit-day-field {
        color: var(--gray);
    }

    input[type="date"]::-webkit-datetime-edit-year-field {
        color: var(--gray);
    }

    /* Методы оплаты */
    .payment-methods {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
        gap: 15px;
        margin-top: 10px;
    }

    .payment-method {
        position: relative;
    }

    .payment-radio {
        display: none;
    }

    .payment-label {
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 10px;
        padding: 20px;
        background: var(--white);
        border: 2px solid var(--light-green);
        border-radius: 12px;
        cursor: pointer;
        transition: all 0.3s ease;
        text-align: center;
        height: 100%;
        justify-content: center;
    }

    .payment-icon {
        font-size: 2rem;
        margin-bottom: 5px;
    }

    .payment-name {
        font-weight: 600;
        color: var(--gray-dark);
        font-size: 0.9rem;
    }

    .payment-desc {
        font-size: 0.8rem;
        color: var(--gray);
        margin-top: 5px;
    }

    .payment-radio:checked + .payment-label {
        border-color: var(--primary-green);
        background: rgba(76, 175, 125, 0.05);
        transform: translateY(-5px);
        box-shadow: 0 10px 20px rgba(76, 175, 125, 0.1);
    }

    .payment-radio:checked + .payment-label .payment-icon {
        transform: scale(1.2);
    }

    /* Кнопка отправки */
    .submit-section {
        margin-top: 40px;
        padding-top: 30px;
        border-top: 2px solid var(--light-green);
    }

    .submit-button {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 15px;
        width: 100%;
        padding: 22px;
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: var(--white);
        border: none;
        border-radius: 15px;
        font-size: 1.2rem;
        font-weight: 700;
        cursor: pointer;
        transition: all 0.4s ease;
        position: relative;
        overflow: hidden;
    }

    .submit-button::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
        transition: left 0.5s ease;
    }

    .submit-button:hover::before {
        left: 100%;
    }

    .submit-button:hover {
        transform: translateY(-5px);
        box-shadow: 0 20px 40px rgba(76, 175, 125, 0.3);
    }

    .submit-button:active {
        transform: translateY(-2px);
    }

    .submit-button::after {
        content: '→';
        font-size: 1.5rem;
        transition: transform 0.3s ease;
    }

    .submit-button:hover::after {
        transform: translateX(10px);
    }

    /* Правая панель */
    .booking-summary {
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.6s forwards;
    }

    .summary-card {
        background: var(--white);
        border-radius: 20px;
        padding: 35px;
        box-shadow: 0 15px 40px rgba(76, 175, 125, 0.1);
        position: sticky;
        top: 100px;
    }

    .summary-title {
        color: var(--dark-green);
        font-size: 1.5rem;
        margin-bottom: 30px;
        text-align: center;
    }

    .summary-details {
        margin-bottom: 30px;
    }

    .summary-item {
        display: flex;
        justify-content: space-between;
        padding: 15px 0;
        border-bottom: 1px solid var(--light-green);
        color: var(--gray-dark);
    }

    .summary-item:last-child {
        border-bottom: none;
    }

    .summary-item.total {
        color: var(--dark-green);
        font-weight: 700;
        font-size: 1.3rem;
        margin-top: 10px;
    }

    .summary-item .value {
        font-weight: 600;
        color: var(--dark-green);
    }

    .summary-note {
        background: rgba(76, 175, 125, 0.1);
        padding: 20px;
        border-radius: 12px;
        margin: 25px 0;
        color: var(--dark-green);
        font-size: 0.9rem;
        line-height: 1.6;
    }

    .summary-note::before {
        content: 'ℹ️';
        margin-right: 10px;
        font-size: 1.2rem;
    }

    /* Безопасность */
    .security-badge {
        display: flex;
        align-items: center;
        gap: 10px;
        color: var(--gray);
        font-size: 0.9rem;
        margin-top: 20px;
        padding-top: 20px;
        border-top: 1px solid var(--light-green);
    }

    .security-badge::before {
        content: '🔒';
        font-size: 1.2rem;
    }

    /* Информационные сообщения */
    .booking-info {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 20px;
        margin-top: 40px;
    }

    .info-card {
        background: var(--white);
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 10px 25px rgba(76, 175, 125, 0.1);
        text-align: center;
        transition: all 0.3s ease;
    }

    .info-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 35px rgba(76, 175, 125, 0.15);
    }

    .info-icon {
        font-size: 2.5rem;
        margin-bottom: 15px;
        display: block;
    }

    .info-card h4 {
        color: var(--dark-green);
        margin-bottom: 10px;
    }

    .info-card p {
        color: var(--gray);
        font-size: 0.9rem;
        line-height: 1.6;
    }

    /* Анимации */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(40px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    /* Валидация */
    .form-input.error,
    .form-select.error,
    .form-textarea.error {
        border-color: var(--error);
        animation: shake 0.5s ease;
    }

    @keyframes shake {
        0%, 100% { transform: translateX(0); }
        25% { transform: translateX(-5px); }
        75% { transform: translateX(5px); }
    }

    .error-message {
        color: var(--error);
        font-size: 0.9rem;
        margin-top: 5px;
        display: flex;
        align-items: center;
        gap: 5px;
    }

    .error-message::before {
        content: '⚠️';
    }

    /* Загрузка */
    .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(255,255,255,0.9);
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s ease;
    }

    .loading-overlay.active {
        opacity: 1;
        visibility: visible;
    }

    .loading-spinner {
        width: 50px;
        height: 50px;
        border: 4px solid var(--light-green);
        border-top-color: var(--primary-green);
        border-radius: 50%;
        animation: spin 1s linear infinite;
        margin-bottom: 20px;
    }

    @keyframes spin {
        to { transform: rotate(360deg); }
    }

    /* Кнопка назад */
    .back-button {
        display: inline-flex;
        align-items: center;
        gap: 10px;
        padding: 12px 25px;
        background: var(--white);
        color: var(--gray-dark);
        text-decoration: none;
        border-radius: 50px;
        font-weight: 500;
        margin-bottom: 30px;
        transition: all 0.3s ease;
        box-shadow: 0 5px 15px rgba(76, 175, 125, 0.1);
    }

    .back-button:hover {
        background: var(--light-green);
        color: var(--dark-green);
        transform: translateX(-5px);
    }

    .back-button::before {
        content: '←';
        transition: transform 0.3s ease;
    }

    .back-button:hover::before {
        transform: translateX(-5px);
    }

    /* Адаптивность */
    @media (max-width: 576px) {
        .booking-title {
            font-size: 2.2rem;
        }

        .form-card {
            padding: 25px;
        }

        .payment-methods {
            grid-template-columns: 1fr;
        }

        .tour-info-card {
            flex-direction: column;
            text-align: center;
        }
    }

    /* Кастомный скроллбар */
    ::-webkit-scrollbar {
        width: 8px;
    }

    ::-webkit-scrollbar-track {
        background: var(--light-green);
        border-radius: 4px;
    }

    ::-webkit-scrollbar-thumb {
        background: var(--primary-green);
        border-radius: 4px;
    }

    ::-webkit-scrollbar-thumb:hover {
        background: var(--dark-green);
    }
</style>

<div class="booking-page">
    <!-- Анимированный фон -->
    <div class="booking-background">
        <div class="booking-shape"></div>
        <div class="booking-shape"></div>
    </div>

    <div class="container booking-container">
        <!-- Кнопка назад -->
        <a href="<?= \yii\helpers\Url::to(['/site/tour', 'id' => $tour->id]) ?>" class="back-button reveal">
            Вернуться к туру
        </a>

        <!-- Заголовок -->
        <div class="booking-header reveal">
            <h1 class="booking-title">Оформление заявки</h1>
            <p class="booking-subtitle">Заполните форму ниже для бронирования тура</p>

            <div class="tour-info-card">
                <?php if ($tour->image): ?>
                    <img src="<?= \yii\helpers\Url::to(['/images/' . $tour->image]) ?>"
                         class="tour-image-small"
                         alt="<?= Html::encode($tour->title) ?>">
                <?php endif; ?>
                <div class="tour-details">
                    <h3><?= Html::encode($tour->title) ?></h3>
                    <p>📅 Длительность: 2 дня</p>
                    <p>👥 Максимум: 12 человек</p>
                    <div class="tour-price-badge">
                        <?= Html::encode($tour->price) ?> ₽
                    </div>
                </div>
            </div>
        </div>

        <!-- Основной контент -->
        <div class="booking-content">
            <!-- Левая часть: Форма -->
            <div class="booking-form-section reveal">
                <div class="form-card">
                    <?php $form = ActiveForm::begin([
                        'id' => 'booking-form',
                        'enableClientValidation' => true,
                    ]); ?>

                    <?= $form->field($model,'tour_id')->hiddenInput(['value'=>$tour->id])->label(false) ?>

                    <!-- Дата тура -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>📅</span>
                            Дата тура
                        </label>
                        <div class="custom-date-input">
                            <?= $form->field($model, 'tour_date', [
                                'template' => '{input}{error}',
                            ])->input('date', [
                                'min' => date('Y-m-d', strtotime('+1 day')),
                                'class' => 'form-input datepicker-input',
                                'required' => true
                            ])->label(false) ?>

                        </div>
                        <div class="hint" style="color: var(--gray); font-size: 0.9rem; margin-top: 5px;">
                            Нажмите на иконку календаря или введите дату в формате ДД.ММ.ГГГГ
                        </div>
                    </div>

                    <!-- Количество человек -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>👥</span>
                            Количество участников
                        </label>
                        <?= $form->field($model, 'persons', [
                            'template' => '{input}{error}',
                            'options' => ['class' => '']
                        ])->input('number', [
                            'min' => 1,
                            'max' => 12,
                            'class' => 'form-input',
                            'required' => true,
                            'placeholder' => 'От 1 до 12 человек'
                        ])->label(false) ?>
                        <div class="counter-controls" style="display: flex; align-items: center; gap: 15px; margin-top: 15px;">
                            <button type="button" class="counter-btn" style="width: 40px; height: 40px; border-radius: 50%; border: 2px solid var(--light-green); background: var(--white); cursor: pointer; font-size: 1.2rem;" onclick="changePersons(-1)">-</button>
                            <span id="persons-display" style="font-weight: 600; color: var(--dark-green);">1</span>
                            <button type="button" class="counter-btn" style="width: 40px; height: 40px; border-radius: 50%; border: 2px solid var(--light-green); background: var(--white); cursor: pointer; font-size: 1.2rem;" onclick="changePersons(1)">+</button>
                        </div>
                    </div>

                    <!-- Особые пожелания -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>💭</span>
                            Особые пожелания
                        </label>
                        <?= $form->field($model, 'wishes', [
                            'template' => '{input}{error}',
                            'options' => ['class' => '']
                        ])->textarea([
                            'rows' => 4,
                            'class' => 'form-textarea',
                            'placeholder' => 'Например: диетическое питание, аллергии, особые требования...'
                        ])->label(false) ?>
                    </div>

                    <!-- Комментарий -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>📝</span>
                            Дополнительный комментарий
                        </label>
                        <?= $form->field($model, 'comment', [
                            'template' => '{input}{error}',
                            'options' => ['class' => '']
                        ])->textarea([
                            'rows' => 3,
                            'class' => 'form-textarea',
                            'placeholder' => 'Любая дополнительная информация...'
                        ])->label(false) ?>
                    </div>

                    <!-- Метод оплаты -->
                    <div class="form-group">
                        <label class="form-label">
                            <span>💳</span>
                            Способ оплаты
                            <span class="required-asterisk">*</span>
                        </label>

                        <div class="payment-methods">
                            <?php
                            $paymentMethods = [
                                'cash' => [
                                    'icon' => '💵',
                                    'name' => 'Наличные',
                                    'desc' => 'Оплата при встрече'
                                ],
                                'card' => [
                                    'icon' => '💳',
                                    'name' => 'Карта онлайн',
                                    'desc' => 'Безопасная оплата'
                                ],
                                'transfer' => [
                                    'icon' => '🏦',
                                    'name' => 'Перевод',
                                    'desc' => 'Банковский счёт'
                                ]
                            ];

                            foreach ($paymentMethods as $value => $data): ?>
                                <div class="payment-method">
                                    <input type="radio"
                                           id="payment_<?= $value ?>"
                                           name="Booking[payment_method]"
                                           value="<?= $value ?>"
                                           class="payment-radio"
                                           required
                                        <?= $model->payment_method == $value ? 'checked' : '' ?>>
                                    <label for="payment_<?= $value ?>" class="payment-label">
                                        <span class="payment-icon"><?= $data['icon'] ?></span>
                                        <span class="payment-name"><?= $data['name'] ?></span>
                                        <span class="payment-desc"><?= $data['desc'] ?></span>
                                    </label>
                                </div>
                            <?php endforeach; ?>
                        </div>

                        <!-- Скрытое поле для валидации Yii -->
                        <?= $form->field($model, 'payment_method', [
                            'template' => '{input}{error}',
                            'options' => ['style' => 'display: none;']
                        ])->hiddenInput(['id' => 'payment-method-hidden'])->label(false) ?>

                        <div class="error-message" id="payment-error" style="display: none; margin-top: 10px;">
                            Пожалуйста, выберите способ оплаты
                        </div>
                    </div>

                    <!-- Кнопка отправки -->
                    <div class="submit-section">
                        <button type="submit" class="submit-button">
                            <span>Подтвердить заявку</span>
                        </button>

                        <div class="terms-agreement" style="text-align: center; margin-top: 20px; color: var(--gray); font-size: 0.9rem;">
                            Нажимая кнопку, вы соглашаетесь с
                            <a href="#" style="color: var(--primary-green); text-decoration: underline;">условиями бронирования</a>
                        </div>
                    </div>

                    <?php ActiveForm::end(); ?>
                </div>

                <!-- Информационные карточки -->
                <div class="booking-info reveal" style="margin-top: 40px;">
                    <div class="info-card">
                        <span class="info-icon">🛡️</span>
                        <h4>Безопасная оплата</h4>
                        <p>Все платежи защищены шифрованием SSL</p>
                    </div>
                    <div class="info-card">
                        <span class="info-icon">↻</span>
                        <h4>Бесплатная отмена</h4>
                        <p>Отмена без штрафа за 48 часов до начала</p>
                    </div>
                    <div class="info-card">
                        <span class="info-icon">📞</span>
                        <h4>Поддержка 24/7</h4>
                        <p>Ответим на все вопросы в любое время</p>
                    </div>
                </div>
            </div>

            <!-- Правая часть: Сводка -->
            <div class="booking-summary reveal">
                <div class="summary-card">
                    <h3 class="summary-title">Сводка заказа</h3>

                    <div class="summary-details">
                        <div class="summary-item">
                            <span>Тур:</span>
                            <span class="value"><?= Html::encode($tour->title) ?></span>
                        </div>
                        <div class="summary-item">
                            <span>Стоимость за человека:</span>
                            <span class="value"><?= Html::encode($tour->price) ?> ₽</span>
                        </div>
                        <div class="summary-item">
                            <span>Количество человек:</span>
                            <span class="value" id="summary-persons">1</span>
                        </div>
                        <div class="summary-item">
                            <span>Дата:</span>
                            <span class="value" id="summary-date">Не выбрана</span>
                        </div>
                        <div class="summary-item total">
                            <span>Итого:</span>
                            <span class="value" id="summary-total"><?= Html::encode($tour->price) ?> ₽</span>
                        </div>
                    </div>

                    <div class="summary-note">
                        После отправки заявки наш менеджер свяжется с вами в течение 2 часов для подтверждения бронирования.
                    </div>

                    <div class="security-badge">
                        Ваши данные защищены
                    </div>
                </div>
            </div>
        </div>

        <!-- Оверлей загрузки -->
        <div class="loading-overlay" id="loadingOverlay">
            <div class="loading-spinner"></div>
            <p style="color: var(--dark-green); font-weight: 600;">Обрабатываем вашу заявку...</p>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Инициализация анимаций
        const reveals = document.querySelectorAll('.reveal');

        function reveal() {
            reveals.forEach(element => {
                const windowHeight = window.innerHeight;
                const elementTop = element.getBoundingClientRect().top;
                const elementVisible = 100;

                if (elementTop < windowHeight - elementVisible) {
                    element.classList.add('active');
                }
            });
        }

        reveal();
        window.addEventListener('scroll', reveal);

        // Обновление сводки
        const personsInput = document.querySelector('[name="Booking[persons]"]');
        const dateInput = document.querySelector('[name="Booking[tour_date]"]');
        const personsDisplay = document.getElementById('persons-display');
        const summaryPersons = document.getElementById('summary-persons');
        const summaryDate = document.getElementById('summary-date');
        const summaryTotal = document.getElementById('summary-total');
        const pricePerPerson = <?= $tour->price ?>;

        function updateSummary() {
            const persons = parseInt(personsInput.value) || 1;
            const date = dateInput.value;
            const total = persons * pricePerPerson;

            personsDisplay.textContent = persons;
            summaryPersons.textContent = persons;
            summaryDate.textContent = date ? formatDate(date) : 'Не выбрана';
            summaryTotal.textContent = total.toLocaleString('ru-RU') + ' ₽';
        }

        function formatDate(dateString) {
            const date = new Date(dateString);
            return date.toLocaleDateString('ru-RU', {
                day: 'numeric',
                month: 'long',
                year: 'numeric'
            });
        }

        // Обработчики событий
        personsInput.addEventListener('input', updateSummary);
        dateInput.addEventListener('change', updateSummary);

        // Кнопки +/- для количества человек
        window.changePersons = function(delta) {
            let currentValue = parseInt(personsInput.value) || 1;
            let newValue = currentValue + delta;

            if (newValue >= 1 && newValue <= 12) {
                personsInput.value = newValue;
                updateSummary();

                // Анимация изменения
                personsDisplay.style.transform = 'scale(1.2)';
                setTimeout(() => {
                    personsDisplay.style.transform = 'scale(1)';
                }, 200);
            }
        }

        // Управление выбором оплаты
        const paymentRadios = document.querySelectorAll('.payment-radio');
        const paymentHidden = document.getElementById('payment-method-hidden');
        const paymentError = document.getElementById('payment-error');

        paymentRadios.forEach(radio => {
            radio.addEventListener('change', function() {
                if (this.checked) {
                    // Обновляем скрытое поле для валидации Yii
                    paymentHidden.value = this.value;

                    // Убираем ошибку если была
                    paymentError.style.display = 'none';

                    // Анимация выбора
                    document.querySelectorAll('.payment-label').forEach(label => {
                        label.style.transform = 'translateY(0)';
                        label.style.boxShadow = 'none';
                    });

                    const label = this.nextElementSibling;
                    label.style.transform = 'translateY(-5px)';
                    label.style.boxShadow = '0 10px 20px rgba(76, 175, 125, 0.1)';

                    // Обновляем иконку
                    const icon = label.querySelector('.payment-icon');
                    icon.style.transform = 'scale(1.2)';

                    // Анимируем другие иконки обратно
                    document.querySelectorAll('.payment-icon').forEach(otherIcon => {
                        if (otherIcon !== icon) {
                            otherIcon.style.transform = 'scale(1)';
                        }
                    });
                }
            });

            // Предзагрузка стилей для выбранного метода
            if (radio.checked) {
                const label = radio.nextElementSibling;
                label.style.transform = 'translateY(-5px)';
                label.style.boxShadow = '0 10px 20px rgba(76, 175, 125, 0.1)';
                label.querySelector('.payment-icon').style.transform = 'scale(1.2)';
            }
        });

        // Валидация формы
        const bookingForm = document.getElementById('booking-form');
        const loadingOverlay = document.getElementById('loadingOverlay');

        bookingForm.addEventListener('submit', function(e) {
            let isValid = true;

            // Проверка выбора оплаты
            const selectedPayment = bookingForm.querySelector('.payment-radio:checked');
            if (!selectedPayment) {
                paymentError.style.display = 'flex';
                isValid = false;

                // Прокрутка к оплате
                document.querySelector('.payment-methods').scrollIntoView({
                    behavior: 'smooth',
                    block: 'center'
                });
            } else {
                paymentError.style.display = 'none';
            }

            // Проверка других обязательных полей
            const requiredFields = bookingForm.querySelectorAll('[required]:not(.payment-radio)');

            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    field.classList.add('error');
                    isValid = false;

                    // Анимация ошибки
                    field.style.animation = 'none';
                    setTimeout(() => {
                        field.style.animation = 'shake 0.5s ease';
                    }, 10);
                } else {
                    field.classList.remove('error');
                }
            });

            if (isValid) {
                loadingOverlay.classList.add('active');

                // Симуляция загрузки (в реальном приложении удалить)
                setTimeout(() => {
                    loadingOverlay.classList.remove('active');
                }, 2000);
            } else {
                e.preventDefault();

                // Прокрутка к первой ошибке
                const firstError = bookingForm.querySelector('.error');
                if (firstError) {
                    firstError.scrollIntoView({
                        behavior: 'smooth',
                        block: 'center'
                    });
                }
            }
        });

        // Удаление класса ошибки при вводе
        bookingForm.querySelectorAll('input, textarea, select').forEach(field => {
            field.addEventListener('input', function() {
                this.classList.remove('error');
            });
        });

        // Удаление ошибки оплаты при выборе
        paymentRadios.forEach(radio => {
            radio.addEventListener('change', function() {
                paymentError.style.display = 'none';
            });
        });

        // Инициализация даты
        const tomorrow = new Date();
        tomorrow.setDate(tomorrow.getDate() + 1);
        const minDate = tomorrow.toISOString().split('T')[0];
        dateInput.min = minDate;

        // Выбор даты по умолчанию через неделю
        const nextWeek = new Date();
        nextWeek.setDate(nextWeek.getDate() + 7);
        dateInput.value = nextWeek.toISOString().split('T')[0];

        updateSummary();

        // Анимация при фокусе
        bookingForm.querySelectorAll('.form-input, .form-select, .form-textarea').forEach(field => {
            field.addEventListener('focus', function() {
                this.parentElement.style.transform = 'translateY(-5px)';
            });

            field.addEventListener('blur', function() {
                this.parentElement.style.transform = 'translateY(0)';
            });
        });

        // Параллакс эффект для фона
        window.addEventListener('scroll', function() {
            const scrolled = window.pageYOffset;
            const background = document.querySelector('.booking-background');
            if (background) {
                background.style.transform = `translateY(${scrolled * 0.05}px)`;
            }
        });

        // Кастомный обработчик для календаря
        dateInput.addEventListener('click', function() {
            this.showPicker ? this.showPicker() : this.type = 'date';
        });

        // Форматирование отображения даты
        function updateDateDisplay() {
            if (dateInput.value) {
                const formatted = formatDate(dateInput.value);
                dateInput.setAttribute('data-formatted', formatted);
            }
        }

        dateInput.addEventListener('change', updateDateDisplay);
        updateDateDisplay();

        // Обработка клавиатуры для календаря
        dateInput.addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                this.blur();
            }
        });
    });
</script>