<?php
use yii\helpers\Html;
$this->title = 'Пользователи';
?>

<style>
    :root {
        --primary-green: #4CAF7D;
        --light-green: #E8F5E9;
        --accent-green: #81C784;
        --dark-green: #388E3C;
        --white: #FFFFFF;
        --gray-light: #F5F5F5;
        --gray: #757575;
        --gray-dark: #424242;
        --warning: #FF9800;
        --error: #f44336;
        --blue: #2196F3;
        --purple: #9C27B0;
        --cyan: #00BCD4;
        --indigo: #3F51B5;
    }

    .users-page {
        min-height: 100vh;
        background: linear-gradient(135deg, var(--light-green) 0%, var(--white) 100%);
        padding: 40px 0;
        position: relative;
        overflow: hidden;
    }

    /* Анимированный фон */
    .users-background {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        z-index: 0;
        opacity: 0.1;
    }

    .users-shape {
        position: absolute;
        border-radius: 50%;
        background: var(--primary-green);
        filter: blur(60px);
        opacity: 0.3;
        animation: float 20s infinite ease-in-out;
    }

    .users-shape:nth-child(1) {
        width: 400px;
        height: 400px;
        top: -100px;
        right: -100px;
        animation-delay: 0s;
    }

    .users-shape:nth-child(2) {
        width: 300px;
        height: 300px;
        bottom: -50px;
        left: -50px;
        animation-delay: -5s;
    }

    .users-shape:nth-child(3) {
        width: 250px;
        height: 250px;
        top: 40%;
        left: 10%;
        animation-delay: -10s;
        background: var(--indigo);
    }

    @keyframes float {
        0%, 100% {
            transform: translate(0, 0) rotate(0deg);
        }
        33% {
            transform: translate(30px, 30px) rotate(120deg);
        }
        66% {
            transform: translate(-20px, 40px) rotate(240deg);
        }
    }

    /* Основной контейнер */
    .users-container {
        position: relative;
        z-index: 1;
        max-width: 1400px;
        margin: 0 auto;
        padding: 0 20px;
    }

    /* Заголовок */
    .users-header {
        text-align: center;
        margin-bottom: 50px;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease forwards;
    }

    .users-title {
        font-size: 3rem;
        font-weight: 800;
        background: linear-gradient(45deg, var(--dark-green), var(--indigo));
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        margin-bottom: 20px;
    }

    .stats-bar {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
        margin-top: 30px;
    }

    .stat-item {
        background: var(--white);
        padding: 20px;
        border-radius: 15px;
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.08);
        display: flex;
        align-items: center;
        gap: 15px;
        transition: all 0.3s ease;
    }

    .stat-item:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 35px rgba(0, 0, 0, 0.12);
    }

    .stat-icon {
        font-size: 2.5rem;
        width: 60px;
        height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
    }

    .stat-icon.total {
        background: rgba(76, 175, 125, 0.1);
        color: var(--primary-green);
    }

    .stat-icon.admin {
        background: rgba(244, 67, 54, 0.1);
        color: var(--error);
    }

    .stat-icon.user {
        background: rgba(33, 150, 243, 0.1);
        color: var(--blue);
    }

    .stat-icon.active {
        background: rgba(156, 39, 176, 0.1);
        color: var(--purple);
    }

    .stat-content h3 {
        font-size: 1.8rem;
        color: var(--dark-green);
        margin: 0;
    }

    .stat-content p {
        color: var(--gray);
        margin: 5px 0 0;
        font-size: 0.9rem;
    }

    /* Поиск и фильтры */
    .filters-section {
        background: var(--white);
        border-radius: 20px;
        padding: 30px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
        margin-bottom: 40px;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.3s forwards;
    }

    .filters-title {
        font-size: 1.3rem;
        color: var(--dark-green);
        margin-bottom: 20px;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .filters-title::before {
        content: '🔍';
        font-size: 1.2rem;
    }

    .search-container {
        display: grid;
        grid-template-columns: 1fr auto auto;
        gap: 15px;
        align-items: end;
    }

    @media (max-width: 768px) {
        .search-container {
            grid-template-columns: 1fr;
        }
    }

    .search-group {
        position: relative;
    }

    .search-input {
        width: 100%;
        padding: 16px 50px 16px 25px;
        border: 2px solid var(--light-green);
        border-radius: 12px;
        font-size: 1rem;
        color: var(--gray-dark);
        background: var(--white);
        transition: all 0.3s ease;
        outline: none;
    }

    .search-input:focus {
        border-color: var(--primary-green);
        box-shadow: 0 0 0 4px rgba(76, 175, 125, 0.1);
    }

    .search-icon {
        position: absolute;
        right: 20px;
        top: 50%;
        transform: translateY(-50%);
        color: var(--gray);
        pointer-events: none;
    }

    .filter-select {
        padding: 16px 20px;
        border: 2px solid var(--light-green);
        border-radius: 12px;
        font-size: 1rem;
        color: var(--gray-dark);
        background: var(--white);
        transition: all 0.3s ease;
        outline: none;
        cursor: pointer;
        appearance: none;
        background-image: url("data:image/svg+xml;charset=UTF-8,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none' stroke='%23757575' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3e%3cpolyline points='6 9 12 15 18 9'%3e%3c/polyline%3e%3c/svg%3e");
        background-repeat: no-repeat;
        background-position: right 20px center;
        background-size: 16px;
    }

    .filter-select:focus {
        border-color: var(--primary-green);
    }

    .search-btn {
        padding: 16px 30px;
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: var(--white);
        border: none;
        border-radius: 12px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .search-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 10px 25px rgba(76, 175, 125, 0.2);
    }

    /* Таблица пользователей */
    .users-table-section {
        background: var(--white);
        border-radius: 20px;
        padding: 40px;
        box-shadow: 0 15px 40px rgba(0, 0, 0, 0.1);
        overflow: hidden;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.6s forwards;
    }

    .table-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 30px;
        flex-wrap: wrap;
        gap: 20px;
    }

    .table-title {
        font-size: 1.8rem;
        color: var(--dark-green);
        margin: 0;
    }

    .export-btn {
        padding: 12px 25px;
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: var(--white);
        border: none;
        border-radius: 10px;
        cursor: pointer;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 8px;
        transition: all 0.3s ease;
        text-decoration: none;
        font-size: 0.95rem;
    }

    .export-btn:hover {
        background: var(--dark-green);
        transform: translateY(-2px);
    }

    /* Таблица */
    .users-table {
        width: 100%;
        border-collapse: separate;
        border-spacing: 0;
        margin: 0;
    }

    .users-table thead th {
        background: linear-gradient(45deg, var(--indigo), #5C6BC0);
        color: var(--white);
        font-weight: 600;
        padding: 20px;
        text-align: left;
        border: none;
        position: sticky;
        top: 0;
        z-index: 10;
    }

    .users-table thead th:first-child {
        border-top-left-radius: 15px;
    }

    .users-table thead th:last-child {
        border-top-right-radius: 15px;
    }

    .users-table tbody tr {
        transition: all 0.3s ease;
        border-bottom: 1px solid var(--light-green);
    }

    .users-table tbody tr:hover {
        background: rgba(76, 175, 125, 0.05);
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
    }

    .users-table tbody td {
        padding: 20px;
        color: var(--gray-dark);
        border: none;
        vertical-align: middle;
    }

    .users-table tbody tr:last-child {
        border-bottom: none;
    }

    /* Аватарки пользователей */
    .user-avatar {
        width: 50px;
        height: 50px;
        border-radius: 50%;
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        display: flex;
        align-items: center;
        justify-content: center;
        color: var(--white);
        font-weight: 700;
        font-size: 1.2rem;
        margin-right: 15px;
        float: left;
    }

    .user-info {
        overflow: hidden;
    }

    .user-name {
        font-weight: 600;
        color: var(--dark-green);
        margin-bottom: 4px;
    }

    .user-email {
        color: var(--gray);
        font-size: 0.9rem;
        margin: 0;
    }

    /* Роли */
    .role-badge {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        padding: 8px 16px;
        border-radius: 50px;
        font-size: 0.9rem;
        font-weight: 600;
        white-space: nowrap;
    }

    .role-admin {
        background: rgba(244, 67, 54, 0.1);
        color: var(--error);
        border: 1px solid rgba(244, 67, 54, 0.2);
    }

    .role-user {
        background: rgba(33, 150, 243, 0.1);
        color: var(--blue);
        border: 1px solid rgba(33, 150, 243, 0.2);
    }

    .role-moderator {
        background: rgba(156, 39, 176, 0.1);
        color: var(--purple);
        border: 1px solid rgba(156, 39, 176, 0.2);
    }

    /* Статусы */
    .status-badge {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        padding: 6px 12px;
        border-radius: 50px;
        font-size: 0.85rem;
        font-weight: 500;
    }

    .status-active {
        background: rgba(76, 175, 125, 0.1);
        color: var(--primary-green);
    }

    .status-inactive {
        background: rgba(158, 158, 158, 0.1);
        color: var(--gray);
    }

    /* Действия */
    .action-buttons {
        display: flex;
        gap: 10px;
        flex-wrap: wrap;
    }

    .action-btn {
        padding: 10px 18px;
        border: none;
        border-radius: 8px;
        font-size: 0.85rem;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 6px;
        text-decoration: none;
        white-space: nowrap;
    }

    .action-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
    }

    .btn-view {
        background: linear-gradient(45deg, var(--blue), #64B5F6);
        color: white;
    }

    .btn-edit {
        background: linear-gradient(45deg, var(--warning), #FFB74D);
        color: white;
    }

    .btn-delete {
        background: linear-gradient(45deg, var(--error), #EF5350);
        color: white;
    }

    .btn-promote {
        background: linear-gradient(45deg, var(--purple), #BA68C8);
        color: white;
    }

    /* Пустое состояние */
    .empty-state {
        text-align: center;
        padding: 80px 20px;
        grid-column: 1 / -1;
    }

    .empty-icon {
        font-size: 5rem;
        color: var(--light-green);
        margin-bottom: 20px;
        opacity: 0.5;
    }

    .empty-state h3 {
        color: var(--gray-dark);
        margin-bottom: 15px;
        font-size: 1.8rem;
    }

    .empty-state p {
        color: var(--gray);
        max-width: 500px;
        margin: 0 auto 30px;
        line-height: 1.6;
    }

    /* Пагинация */
    .pagination-section {
        margin-top: 40px;
        display: flex;
        justify-content: center;
        opacity: 0;
        transform: translateY(20px);
        animation: fadeInUp 0.8s ease 0.9s forwards;
    }

    .pagination {
        display: flex;
        gap: 8px;
        list-style: none;
        padding: 0;
        margin: 0;
    }

    .page-item {
        min-width: 40px;
    }

    .page-link {
        display: flex;
        align-items: center;
        justify-content: center;
        height: 40px;
        padding: 0 15px;
        background: var(--white);
        border: 2px solid var(--light-green);
        border-radius: 10px;
        color: var(--gray-dark);
        text-decoration: none;
        font-weight: 600;
        transition: all 0.3s ease;
    }

    .page-link:hover {
        background: var(--primary-green);
        color: var(--white);
        border-color: var(--primary-green);
        transform: translateY(-2px);
    }

    .page-item.active .page-link {
        background: var(--primary-green);
        color: var(--white);
        border-color: var(--primary-green);
    }

    /* Анимации */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(40px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    /* Кастомный скроллбар для таблицы */
    .table-wrapper {
        overflow-x: auto;
        border-radius: 15px;
        margin: 0 -10px;
        padding: 10px;
    }

    .table-wrapper::-webkit-scrollbar {
        height: 8px;
    }

    .table-wrapper::-webkit-scrollbar-track {
        background: var(--light-green);
        border-radius: 4px;
    }

    .table-wrapper::-webkit-scrollbar-thumb {
        background: var(--primary-green);
        border-radius: 4px;
    }

    .table-wrapper::-webkit-scrollbar-thumb:hover {
        background: var(--dark-green);
    }

    /* Модальное окно */
    .modal-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.5);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s ease;
    }

    .modal-overlay.active {
        opacity: 1;
        visibility: visible;
    }

    .modal-content {
        background: var(--white);
        border-radius: 20px;
        padding: 40px;
        max-width: 400px;
        width: 90%;
        transform: translateY(30px);
        transition: transform 0.3s ease;
    }

    .modal-overlay.active .modal-content {
        transform: translateY(0);
    }

    /* Адаптивность */
    @media (max-width: 1200px) {
        .users-container {
            padding: 0 15px;
        }

        .users-table-section {
            padding: 25px;
        }
    }

    @media (max-width: 768px) {
        .users-title {
            font-size: 2.2rem;
        }

        .table-header {
            flex-direction: column;
            align-items: stretch;
        }

        .action-buttons {
            flex-direction: column;
        }

        .action-btn {
            width: 100%;
            justify-content: center;
        }

        .stats-bar {
            grid-template-columns: 1fr;
        }

        .users-table {
            font-size: 0.9rem;
        }

        .users-table td,
        .users-table th {
            padding: 12px 8px;
        }
    }

    @media (max-width: 576px) {
        .users-table-section {
            padding: 20px;
        }

        .filters-section {
            padding: 20px;
        }

        .users-title {
            font-size: 1.8rem;
        }

        .user-avatar {
            float: none;
            margin: 0 auto 10px;
        }

        .user-info {
            text-align: center;
        }
    }

    /* Кнопка назад */
    .back-button {
        display: inline-flex;
        align-items: center;
        gap: 10px;
        padding: 12px 25px;
        background: var(--white);
        color: var(--gray-dark);
        text-decoration: none;
        border-radius: 50px;
        font-weight: 500;
        margin-bottom: 30px;
        transition: all 0.3s ease;
        box-shadow: 0 5px 15px rgba(76, 175, 125, 0.1);
    }

    .back-button:hover {
        background: var(--light-green);
        color: var(--dark-green);
        transform: translateX(-5px);
    }

    .back-button::before {
        content: '←';
        transition: transform 0.3s ease;
    }

    .back-button:hover::before {
        transform: translateX(-5px);
    }

    /* Блок с информацией о пользователе */
    .user-details {
        background: var(--light-green);
        padding: 15px;
        border-radius: 10px;
        margin-top: 10px;
        display: none;
    }

    .user-details.active {
        display: block;
        animation: fadeIn 0.3s ease;
    }

    @keyframes fadeIn {
        from { opacity: 0; }
        to { opacity: 1; }
    }

    /* Загрузка */
    .loading {
        display: inline-block;
        width: 20px;
        height: 20px;
        border: 3px solid var(--light-green);
        border-top-color: var(--primary-green);
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }

    @keyframes spin {
        to { transform: rotate(360deg); }
    }
</style>

<div class="users-page">
    <!-- Анимированный фон -->
    <div class="users-background">
        <div class="users-shape"></div>
        <div class="users-shape"></div>
        <div class="users-shape"></div>
    </div>

    <div class="container users-container">
        <!-- Кнопка назад -->
        <a href="<?= \yii\helpers\Url::to(['admin/index']) ?>" class="back-button reveal">
            Назад в админ-панель
        </a>

        <!-- Заголовок -->
        <div class="users-header reveal">
            <h1 class="users-title">Управление пользователями</h1>

        </div>



        <!-- Таблица пользователей -->
        <div class="users-table-section reveal">
            <div class="table-header">
                <h2 class="table-title">Список пользователей</h2>
                <div class="table-controls">

                </div>
            </div>

            <div class="table-wrapper">
                <table class="users-table">
                    <thead>
                    <tr>
                        <th>Пользователь</th>
                        <th>Контактные данные</th>
                        <th>Роль</th>

                        <th>Действия</th>
                    </tr>
                    </thead>
                    <tbody id="usersTableBody">
                    <?php if (empty($users)): ?>
                        <tr>
                            <td colspan="5" class="empty-state">
                                <div class="empty-icon">👥</div>
                                <h3>Пользователи не найдены</h3>
                                <p>В системе пока нет зарегистрированных пользователей</p>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach($users as $user): ?>
                            <tr data-user-id="<?= $user->id ?>"
                                data-role="<?= strtolower($user->role) ?>"
                                data-status="active">
                                <!-- Аватар и имя -->
                                <td>
                                    <div class="user-avatar">
                                        <?= mb_substr($user->full_name, 0, 1) ?>
                                    </div>
                                    <div class="user-info">
                                        <div class="user-name"><?= Html::encode($user->full_name) ?></div>
                                        <div class="user-id">ID: <?= $user->id ?></div>
                                    </div>
                                </td>

                                <!-- Контактные данные -->
                                <td>
                                    <div style="margin-bottom: 8px;">
                                        <strong>📧 Email:</strong>
                                        <div style="color: var(--gray-dark);"><?= Html::encode($user->email) ?></div>
                                    </div>
                                    <div>
                                        <strong>📱 Телефон:</strong>
                                        <div style="color: var(--gray-dark);"><?= Html::encode($user->phone) ?></div>
                                    </div>
                                    <button class="toggle-details"
                                            onclick="toggleUserDetails(<?= $user->id ?>)"
                                            style="
                                                    background: none;
                                                    border: none;
                                                    color: var(--primary-green);
                                                    cursor: pointer;
                                                    font-size: 0.85rem;
                                                    margin-top: 8px;
                                                    display: flex;
                                                    align-items: center;
                                                    gap: 5px;
                                                ">
                                    </button>
                                    <div class="user-details" id="details-<?= $user->id ?>">
                                        <div style="font-size: 0.85rem; color: var(--gray);">
                                            <div><strong>Адрес:</strong> <?= Html::encode($user->address ?? 'Не указан') ?></div>
                                            <div><strong>Паспорт:</strong> <?= Html::encode($user->passport ?? 'Не указан') ?></div>
                                            <div><strong>Последний вход:</strong> <?= date('d.m.Y H:i') ?></div>
                                        </div>
                                    </div>
                                </td>

                                <!-- Роль и статус -->
                                <td>
                                    <div style="margin-bottom: 10px;">
                                        <?php
                                        $roleClass = '';
                                        if (strtolower($user->role) == 'admin') $roleClass = 'role-admin';
                                        elseif (strtolower($user->role) == 'user') $roleClass = 'role-user';
                                        elseif (strtolower($user->role) == 'moderator') $roleClass = 'role-moderator';
                                        ?>
                                        <div class="role-badge <?= $roleClass ?>">
                                            <?php if (strtolower($user->role) == 'admin'): ?>👑
                                            <?php elseif (strtolower($user->role) == 'user'): ?>👤
                                            <?php elseif (strtolower($user->role) == 'moderator'): ?>🛡️
                                            <?php endif; ?>
                                            <?= Html::encode($user->role) ?>
                                        </div>
                                    </div>
                                    <div class="status-badge status-active">
                                    </div>
                                    <div style="margin-top: 8px; font-size: 0.85rem; color: var(--gray);">

                                    </div>
                                </td>


                                <!-- Действия -->
                                <td>


                                        <?= Html::a('🗑️ Удалить', ['delete-user', 'id' => $user->id], [
                                            'class' => 'action-btn btn-delete',
                                            'title' => 'Удалить пользователя',
                                            'data-method' => 'post',
                                            'onclick' => 'return confirmDelete(this, ' . $user->id . ');'
                                        ]) ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Пагинация (если есть) -->
            <?php if (isset($pagination)): ?>
                <div class="pagination-section">
                    <ul class="pagination">
                        <?php if ($pagination->page > 0): ?>
                            <li class="page-item">
                                <a class="page-link" href="<?= \yii\helpers\Url::current(['page' => $pagination->page]) ?>">
                                    ← Назад
                                </a>
                            </li>
                        <?php endif; ?>

                        <?php for ($i = 1; $i <= $pagination->pageCount; $i++): ?>
                            <li class="page-item <?= ($i == $pagination->page + 1) ? 'active' : '' ?>">
                                <a class="page-link" href="<?= \yii\helpers\Url::current(['page' => $i - 1]) ?>">
                                    <?= $i ?>
                                </a>
                            </li>
                        <?php endfor; ?>

                        <?php if ($pagination->page < $pagination->pageCount - 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="<?= \yii\helpers\Url::current(['page' => $pagination->page + 2]) ?>">
                                    Вперед →
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </div>
            <?php endif; ?>
        </div>

        <!-- Модальное окно подтверждения -->
        <div class="modal-overlay" id="confirmModal">
            <div class="modal-content">
                <div style="text-align: center;">
                    <div style="font-size: 4rem; color: var(--error); margin-bottom: 20px;">🗑️</div>
                    <h3 style="color: var(--dark-green); margin-bottom: 15px;">Подтверждение удаления</h3>
                    <p id="confirmText" style="color: var(--gray); margin-bottom: 30px; line-height: 1.6;">
                        Вы уверены, что хотите удалить этого пользователя? Все его данные будут безвозвратно удалены.
                    </p>
                    <div style="display: flex; gap: 15px; justify-content: center;">
                        <button class="confirm-btn cancel" onclick="closeConfirmModal()"
                                style="padding: 12px 30px; background: var(--light-green); color: var(--gray-dark); border: none; border-radius: 10px; font-weight: 600; cursor: pointer; transition: all 0.3s ease;">
                            Отмена
                        </button>
                        <button class="confirm-btn delete" id="confirmDeleteBtn"
                                style="padding: 12px 30px; background: var(--error); color: white; border: none; border-radius: 10px; font-weight: 600; cursor: pointer; transition: all 0.3s ease;">
                            Удалить
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Инициализация анимаций
        const reveals = document.querySelectorAll('.reveal');

        function reveal() {
            reveals.forEach(element => {
                const windowHeight = window.innerHeight;
                const elementTop = element.getBoundingClientRect().top;
                const elementVisible = 100;

                if (elementTop < windowHeight - elementVisible) {
                    element.classList.add('active');
                }
            });
        }

        reveal();
        window.addEventListener('scroll', reveal);

        // Подсчет статистики
        function calculateStats() {
            const rows = document.querySelectorAll('#usersTableBody tr[data-role]');
            let adminCount = 0;
            let userCount = 0;

            rows.forEach(row => {
                const role = row.getAttribute('data-role');
                if (role === 'admin') adminCount++;
                if (role === 'user') userCount++;
            });

            // Анимируем изменение чисел
            animateCounter('admin-count', adminCount);
            animateCounter('user-count', userCount);
            animateCounter('total-users', rows.length);
        }

        // Анимация счетчика
        function animateCounter(elementId, targetValue) {
            const element = document.getElementById(elementId);
            if (!element) return;

            let current = parseInt(element.textContent) || 0;
            const increment = targetValue > current ? 1 : -1;
            const stepTime = Math.abs(50 / (targetValue - current));

            function updateCounter() {
                current += increment;
                element.textContent = current;

                if (current !== targetValue) {
                    setTimeout(updateCounter, stepTime);
                }
            }

            updateCounter();
        }

        // Запускаем подсчет статистики
        calculateStats();

        // Параллакс эффект для фона
        window.addEventListener('scroll', function() {
            const scrolled = window.pageYOffset;
            const background = document.querySelector('.users-background');
            if (background) {
                background.style.transform = `translateY(${scrolled * 0.05}px)`;
            }
        });

        // Поиск пользователей
        window.searchUsers = function() {
            const searchText = document.getElementById('searchInput').value.toLowerCase();
            const roleFilter = document.getElementById('roleFilter').value;
            const rows = document.querySelectorAll('#usersTableBody tr');

            rows.forEach(row => {
                const name = row.querySelector('.user-name').textContent.toLowerCase();
                const email = row.querySelector('td:nth-child(2)').textContent.toLowerCase();
                const role = row.getAttribute('data-role');

                let matchesSearch = true;
                let matchesRole = true;

                // Проверка поиска
                if (searchText && !name.includes(searchText) && !email.includes(searchText)) {
                    matchesSearch = false;
                }

                // Проверка роли
                if (roleFilter && role !== roleFilter) {
                    matchesRole = false;
                }

                if (matchesSearch && matchesRole) {
                    row.style.display = '';
                    setTimeout(() => {
                        row.style.opacity = '1';
                        row.style.transform = 'translateY(0)';
                    }, 10);
                } else {
                    row.style.opacity = '0';
                    row.style.transform = 'translateY(20px)';
                    setTimeout(() => {
                        row.style.display = 'none';
                    }, 300);
                }
            });

            // Обновляем статистику
            setTimeout(calculateStats, 350);
        };

        // Обновление пользователей
        window.refreshUsers = function() {
            const refreshBtn = document.querySelector('.refresh-btn');
            refreshBtn.innerHTML = '<div class="loading"></div>';
            refreshBtn.disabled = true;

            // Симуляция обновления
            setTimeout(() => {
                refreshBtn.innerHTML = '🔄';
                refreshBtn.disabled = false;

                showNotification('Обновлено', 'Список пользователей обновлен', 'success');
            }, 1500);
        };

        // Экспорт пользователей
        window.exportUsers = function() {
            const rows = document.querySelectorAll('#usersTableBody tr:not(.empty-state)');
            if (rows.length === 0) {
                alert('Нет данных для экспорта');
                return;
            }

            let csvContent = "data:text/csv;charset=utf-8,";
            csvContent += "ID,Имя,Email,Телефон,Роль,Дата регистрации\n";

            rows.forEach(row => {
                if (row.style.display !== 'none') {
                    const id = row.getAttribute('data-user-id');
                    const name = row.querySelector('.user-name').textContent;
                    const email = row.querySelector('td:nth-child(2) div:nth-child(2)').textContent.trim();
                    const phone = row.querySelector('td:nth-child(2) div:nth-child(4)').textContent.trim();
                    const role = row.querySelector('.role-badge').textContent.trim();
                    const date = row.querySelector('td:nth-child(4) div:nth-child(1)').textContent.trim();

                    csvContent += `${id},"${name}","${email}","${phone}","${role}","${date}"\n`;
                }
            });

            const encodedUri = encodeURI(csvContent);
            const link = document.createElement("a");
            link.setAttribute("href", encodedUri);
            link.setAttribute("download", "users_" + new Date().toISOString().slice(0, 10) + ".csv");
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);

            showNotification('Экспорт завершен', 'Файл успешно скачан', 'success');
        };

        // Управление деталями пользователя
        window.toggleUserDetails = function(userId) {
            const details = document.getElementById('details-' + userId);
            const button = document.querySelector(`[onclick="toggleUserDetails(${userId})"]`);

            if (details.classList.contains('active')) {
                details.classList.remove('active');
                button.innerHTML = '<span>📋</span> Подробнее';
            } else {
                // Закрываем все открытые детали
                document.querySelectorAll('.user-details.active').forEach(d => {
                    d.classList.remove('active');
                });

                // Обновляем все кнопки
                document.querySelectorAll('.toggle-details').forEach(btn => {
                    btn.innerHTML = '<span>📋</span> Подробнее';
                });

                details.classList.add('active');
                button.innerHTML = '<span>📋</span> Скрыть';
            }
        };

        // Просмотр пользователя
        window.viewUser = function(userId) {
            showNotification('Просмотр пользователя', 'Открывается профиль пользователя ID: ' + userId, 'info');
            // В реальном приложении: window.location.href = '/admin/user/' + userId;
        };

        // Редактирование пользователя
        window.editUser = function(userId) {
            showNotification('Редактирование', 'Открывается форма редактирования пользователя', 'info');
            // В реальном приложении: window.location.href = '/admin/user/edit/' + userId;
        };

        // Назначение администратором
        window.promoteUser = function(userId) {
            if (confirm('Вы уверены, что хотите назначить этого пользователя администратором?')) {
                const row = document.querySelector(`tr[data-user-id="${userId}"]`);
                const roleBadge = row.querySelector('.role-badge');

                roleBadge.className = 'role-badge role-admin';
                roleBadge.innerHTML = '👑 Администратор';
                row.setAttribute('data-role', 'admin');

                // Убираем кнопку назначения
                const promoteBtn = row.querySelector('.btn-promote');
                if (promoteBtn) promoteBtn.remove();

                calculateStats();
                showNotification('Пользователь повышен', 'Пользователь назначен администратором', 'success');
            }
        };

        // Подтверждение удаления
        let deleteUrl = '';
        let deleteUserId = '';

        window.confirmDelete = function(element, userId) {
            event.preventDefault();
            deleteUrl = element.href;
            deleteUserId = userId;

            const modal = document.getElementById('confirmModal');
            const userName = document.querySelector(`tr[data-user-id="${userId}"] .user-name`).textContent;
            document.getElementById('confirmText').textContent =
                `Вы уверены, что хотите удалить пользователя "${userName}"? Все его данные будут безвозвратно удалены.`;

            modal.classList.add('active');
            document.body.style.overflow = 'hidden';

            return false;
        };

        window.closeConfirmModal = function() {
            const modal = document.getElementById('confirmModal');
            modal.classList.remove('active');
            document.body.style.overflow = '';
        };

        // Обработчик удаления
        document.getElementById('confirmDeleteBtn').addEventListener('click', function() {
            const row = document.querySelector(`tr[data-user-id="${deleteUserId}"]`);

            // Анимация удаления
            row.style.opacity = '0';
            row.style.transform = 'translateX(100px) scale(0.8)';

            setTimeout(() => {
                row.remove();
                closeConfirmModal();
                calculateStats();

                showNotification('Пользователь удален', 'Пользователь успешно удален из системы', 'success');

                // Проверяем пустое состояние
                const tableBody = document.getElementById('usersTableBody');
                if (tableBody.children.length === 0 ||
                    (tableBody.children.length === 1 && tableBody.children[0].classList.contains('empty-state'))) {
                    const emptyState = document.createElement('tr');
                    emptyState.innerHTML = `
                        <td colspan="5" class="empty-state">
                            <div class="empty-icon">👥</div>
                            <h3>Пользователи не найдены</h3>
                            <p>В системе пока нет зарегистрированных пользователей</p>
                        </td>
                    `;
                    tableBody.appendChild(emptyState);
                }
            }, 300);
        });

        // Функция показа уведомлений
        function showNotification(title, message, type = 'success') {
            const notification = document.createElement('div');
            notification.className = 'notification';
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: white;
                padding: 15px 25px;
                border-radius: 10px;
                box-shadow: 0 10px 30px rgba(0,0,0,0.15);
                border-left: 4px solid var(--${type === 'success' ? 'primary-green' : type === 'error' ? 'error' : 'blue'});
                z-index: 1000;
                animation: slideInRight 0.3s ease;
                max-width: 350px;
            `;

            notification.innerHTML = `
                <strong>${title}</strong><br>
                <small>${message}</small>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                notification.style.opacity = '0';
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 5000);
        }

        // Добавляем CSS для анимации уведомлений
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideInRight {
                from {
                    transform: translateX(100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
        `;
        document.head.appendChild(style);

        // Поиск по мере ввода
        const searchInput = document.getElementById('searchInput');
        let searchTimeout;

        searchInput.addEventListener('input', function() {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(searchUsers, 500);
        });

        // Сортировка таблицы
        document.querySelectorAll('.users-table th').forEach((th, index) => {
            th.style.cursor = 'pointer';
            th.addEventListener('click', function() {
                const table = this.closest('table');
                const tbody = table.querySelector('tbody');
                const rows = Array.from(tbody.querySelectorAll('tr:not(.empty-state)'));

                if (rows.length === 0) return;

                // Определяем тип сортировки
                let sortFunction;
                switch(index) {
                    case 0: // Имя
                        sortFunction = (a, b) =>
                            a.querySelector('.user-name').textContent
                                .localeCompare(b.querySelector('.user-name').textContent);
                        break;
                    case 1: // Email
                        sortFunction = (a, b) => {
                            const aEmail = a.querySelector('td:nth-child(2) div:nth-child(2)').textContent.trim();
                            const bEmail = b.querySelector('td:nth-child(2) div:nth-child(2)').textContent.trim();
                            return aEmail.localeCompare(bEmail);
                        };
                        break;
                    case 2: // Роль
                        sortFunction = (a, b) => {
                            const aRole = a.getAttribute('data-role');
                            const bRole = b.getAttribute('data-role');
                            const roleOrder = {admin: 1, moderator: 2, user: 3};
                            return (roleOrder[aRole] || 4) - (roleOrder[bRole] || 4);
                        };
                        break;
                    case 3: // Дата регистрации
                        sortFunction = (a, b) => {
                            const aDate = a.querySelector('td:nth-child(4) div:nth-child(1)').textContent.trim();
                            const bDate = b.querySelector('td:nth-child(4) div:nth-child(1)').textContent.trim();
                            return new Date(aDate.split('.').reverse().join('-')) -
                                new Date(bDate.split('.').reverse().join('-'));
                        };
                        break;
                    default:
                        return;
                }

                // Сортировка
                rows.sort(sortFunction);

                // Пересобираем таблицу
                rows.forEach((row, i) => {
                    tbody.appendChild(row);

                    // Анимация
                    setTimeout(() => {
                        row.style.opacity = '0.5';
                        row.style.transform = 'translateY(10px)';
                        setTimeout(() => {
                            row.style.opacity = '1';
                            row.style.transform = 'translateY(0)';
                        }, 100);
                    }, i * 50);
                });
            });
        });

        // Эффекты при наведении на строки
        document.querySelectorAll('.users-table tbody tr').forEach(row => {
            row.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-2px)';
            });

            row.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
            });
        });

        // Инициализация
        calculateStats();
    });
</script>