<?php
use yii\helpers\Html;
use yii\helpers\Url;

$this->title = 'Управление турами';
?>

<style>
    :root {
        --primary-green: #4CAF7D;
        --light-green: #E8F5E9;
        --accent-green: #81C784;
        --dark-green: #388E3C;
        --white: #FFFFFF;
        --gray-light: #F5F5F5;
        --gray: #757575;
        --gray-dark: #424242;
        --warning: #FF9800;
        --error: #f44336;
        --blue: #2196F3;
        --purple: #9C27B0;
        --cyan: #00BCD4;
    }

    .tours-page {
        min-height: 100vh;
        background: linear-gradient(135deg, var(--light-green) 0%, var(--white) 100%);
        padding: 40px 0;
        position: relative;
        overflow: hidden;
    }

    /* Анимированный фон */
    .tours-background {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        z-index: 0;
        opacity: 0.1;
    }

    .tours-shape {
        position: absolute;
        border-radius: 50%;
        background: var(--primary-green);
        filter: blur(60px);
        opacity: 0.3;
        animation: float 20s infinite ease-in-out;
    }

    .tours-shape:nth-child(1) {
        width: 400px;
        height: 400px;
        top: -100px;
        right: -100px;
        animation-delay: 0s;
    }

    .tours-shape:nth-child(2) {
        width: 300px;
        height: 300px;
        bottom: -50px;
        left: -50px;
        animation-delay: -5s;
    }

    .tours-shape:nth-child(3) {
        width: 200px;
        height: 200px;
        top: 40%;
        left: 10%;
        animation-delay: -10s;
        background: var(--warning);
    }

    @keyframes float {
        0%, 100% {
            transform: translate(0, 0) rotate(0deg);
        }
        33% {
            transform: translate(30px, 30px) rotate(120deg);
        }
        66% {
            transform: translate(-20px, 40px) rotate(240deg);
        }
    }

    /* Основной контейнер */
    .tours-container {
        position: relative;
        z-index: 1;
        max-width: 1600px;
        margin: 0 auto;
        padding: 0 20px;
    }

    /* Заголовок */
    .tours-header {
        text-align: center;
        margin-bottom: 50px;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease forwards;
    }

    .tours-title {
        font-size: 3rem;
        font-weight: 800;
        background: linear-gradient(45deg, var(--dark-green), var(--warning));
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        margin-bottom: 20px;
    }

    .header-actions {
        display: flex;
        justify-content: space-between;
        align-items: center;
        flex-wrap: wrap;
        gap: 20px;
        margin-top: 30px;
    }

    .create-tour-btn {
        display: inline-flex;
        align-items: center;
        gap: 12px;
        padding: 16px 35px;
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: var(--white);
        text-decoration: none;
        border-radius: 50px;
        font-weight: 700;
        font-size: 1.1rem;
        transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        box-shadow: 0 10px 25px rgba(76, 175, 125, 0.2);
        border: none;
        cursor: pointer;
    }

    .create-tour-btn:hover {
        transform: translateY(-5px) scale(1.05);
        box-shadow: 0 20px 40px rgba(76, 175, 125, 0.3);
    }

    .create-tour-btn::before {
        content: '+';
        font-size: 1.5rem;
        font-weight: 300;
    }

    .header-stats {
        display: flex;
        gap: 25px;
        flex-wrap: wrap;
    }

    .stat-badge {
        background: var(--white);
        padding: 12px 25px;
        border-radius: 50px;
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        display: flex;
        align-items: center;
        gap: 10px;
        font-weight: 600;
        color: var(--gray-dark);
        transition: all 0.3s ease;
    }

    .stat-badge:hover {
        transform: translateY(-3px);
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.12);
    }

    .stat-badge .count {
        background: var(--light-green);
        color: var(--dark-green);
        width: 30px;
        height: 30px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: 700;
    }

    /* Карточки туров */
    .tours-grid-section {
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.3s forwards;
    }

    .tours-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(500px, 1fr));
        gap: 30px;
    }

    @media (max-width: 1200px) {
        .tours-grid {
            grid-template-columns: repeat(auto-fill, minmax(400px, 1fr));
        }
    }

    @media (max-width: 768px) {
        .tours-grid {
            grid-template-columns: 1fr;
        }
    }

    .tour-card {
        background: var(--white);
        border-radius: 20px;
        overflow: hidden;
        box-shadow: 0 15px 35px rgba(0, 0, 0, 0.1);
        transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        position: relative;
        border: 2px solid transparent;
    }

    .tour-card:hover {
        transform: translateY(-10px) scale(1.02);
        box-shadow: 0 25px 50px rgba(0, 0, 0, 0.15);
        border-color: var(--primary-green);
    }

    .tour-image-container {
        height: 250px;
        overflow: hidden;
        position: relative;
    }

    .tour-image {
        width: 100%;
        height: 100%;
        object-fit: cover;
        transition: transform 0.6s ease;
    }

    .tour-card:hover .tour-image {
        transform: scale(1.1);
    }

    .tour-badge {
        position: absolute;
        top: 20px;
        right: 20px;
        background: linear-gradient(45deg, var(--warning), #FFB74D);
        color: var(--white);
        padding: 8px 20px;
        border-radius: 50px;
        font-weight: 700;
        font-size: 1.2rem;
        box-shadow: 0 5px 15px rgba(255, 152, 0, 0.3);
        z-index: 2;
    }

    .tour-content {
        padding: 30px;
    }

    .tour-header {
        margin-bottom: 20px;
    }

    .tour-title {
        font-size: 1.5rem;
        color: var(--dark-green);
        margin: 0 0 10px 0;
        font-weight: 700;
        line-height: 1.3;
    }

    .tour-description {
        color: var(--gray);
        line-height: 1.6;
        margin-bottom: 20px;
        display: -webkit-box;
        -webkit-line-clamp: 3;
        -webkit-box-orient: vertical;
        overflow: hidden;
    }

    .tour-details {
        background: var(--light-green);
        padding: 20px;
        border-radius: 15px;
        margin-bottom: 25px;
    }

    .detail-item {
        display: flex;
        align-items: center;
        gap: 10px;
        margin-bottom: 10px;
        color: var(--gray-dark);
    }

    .detail-item:last-child {
        margin-bottom: 0;
    }

    .detail-icon {
        color: var(--primary-green);
        font-size: 1.2rem;
        width: 24px;
    }

    .tour-program {
        margin-bottom: 25px;
    }

    .program-title {
        font-weight: 600;
        color: var(--dark-green);
        margin-bottom: 10px;
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .program-title::before {
        content: '📋';
    }

    .program-text {
        color: var(--gray);
        line-height: 1.6;
        font-size: 0.95rem;
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
    }

    .tour-actions {
        display: flex;
        gap: 12px;
        margin-top: 25px;
    }

    .action-btn {
        flex: 1;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
        padding: 14px;
        border-radius: 12px;
        font-weight: 600;
        font-size: 0.95rem;
        transition: all 0.3s ease;
        text-decoration: none;
        border: none;
        cursor: pointer;
    }

    .action-btn:hover {
        transform: translateY(-3px);
        box-shadow: 0 8px 20px rgba(0, 0, 0, 0.1);
    }

    .btn-edit {
        background: linear-gradient(45deg, var(--warning), #FFB74D);
        color: var(--white);
    }

    .btn-delete {
        background: linear-gradient(45deg, var(--error), #EF5350);
        color: var(--white);
    }

    .btn-view {
        background: linear-gradient(45deg, var(--blue), #64B5F6);
        color: var(--white);
    }

    /* Пустое состояние */
    .empty-state {
        text-align: center;
        padding: 80px 20px;
        background: var(--white);
        border-radius: 20px;
        box-shadow: 0 15px 35px rgba(0, 0, 0, 0.1);
        grid-column: 1 / -1;
    }

    .empty-icon {
        font-size: 5rem;
        color: var(--light-green);
        margin-bottom: 20px;
        opacity: 0.5;
    }

    .empty-state h3 {
        color: var(--gray-dark);
        margin-bottom: 15px;
        font-size: 1.8rem;
    }

    .empty-state p {
        color: var(--gray);
        max-width: 500px;
        margin: 0 auto 30px;
        line-height: 1.6;
    }

    /* Фильтры */
    .filters-section {
        background: var(--white);
        border-radius: 20px;
        padding: 30px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
        margin-bottom: 40px;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.6s forwards;
    }

    .filters-title {
        font-size: 1.3rem;
        color: var(--dark-green);
        margin-bottom: 20px;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .filters-title::before {
        content: '🔍';
    }

    .filter-form {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
    }

    .filter-group {
        flex: 1;
    }

    .filter-label {
        display: block;
        margin-bottom: 8px;
        color: var(--gray-dark);
        font-weight: 600;
        font-size: 0.95rem;
    }

    .filter-input {
        width: 100%;
        padding: 14px 20px;
        border: 2px solid var(--light-green);
        border-radius: 12px;
        font-size: 1rem;
        color: var(--gray-dark);
        background: var(--white);
        transition: all 0.3s ease;
        outline: none;
    }

    .filter-input:focus {
        border-color: var(--primary-green);
        box-shadow: 0 0 0 4px rgba(76, 175, 125, 0.1);
    }

    .filter-btn {
        align-self: end;
        padding: 14px 25px;
        background: var(--primary-green);
        color: var(--white);
        border: none;
        border-radius: 12px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
    }

    .filter-btn:hover {
        background: var(--dark-green);
        transform: translateY(-2px);
    }

    /* Анимации */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(40px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    /* Кастомный скроллбар */
    ::-webkit-scrollbar {
        width: 8px;
    }

    ::-webkit-scrollbar-track {
        background: var(--light-green);
        border-radius: 4px;
    }

    ::-webkit-scrollbar-thumb {
        background: var(--primary-green);
        border-radius: 4px;
    }

    ::-webkit-scrollbar-thumb:hover {
        background: var(--dark-green);
    }

    /* Модальное окно подтверждения */
    .confirm-modal {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.5);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s ease;
    }

    .confirm-modal.active {
        opacity: 1;
        visibility: visible;
    }

    .confirm-content {
        background: var(--white);
        border-radius: 20px;
        padding: 40px;
        max-width: 400px;
        width: 90%;
        text-align: center;
        transform: translateY(30px);
        transition: transform 0.3s ease;
    }

    .confirm-modal.active .confirm-content {
        transform: translateY(0);
    }

    .confirm-icon {
        font-size: 4rem;
        color: var(--error);
        margin-bottom: 20px;
    }

    .confirm-text {
        color: var(--gray-dark);
        margin-bottom: 30px;
        line-height: 1.6;
    }

    .confirm-actions {
        display: flex;
        gap: 15px;
        justify-content: center;
    }

    .confirm-btn {
        padding: 12px 30px;
        border: none;
        border-radius: 10px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
    }

    .confirm-btn.cancel {
        background: var(--light-green);
        color: var(--gray-dark);
    }

    .confirm-btn.delete {
        background: var(--error);
        color: var(--white);
    }

    /* Кнопка назад */
    .back-button {
        display: inline-flex;
        align-items: center;
        gap: 10px;
        padding: 12px 25px;
        background: var(--white);
        color: var(--gray-dark);
        text-decoration: none;
        border-radius: 50px;
        font-weight: 500;
        margin-bottom: 30px;
        transition: all 0.3s ease;
        box-shadow: 0 5px 15px rgba(76, 175, 125, 0.1);
    }

    .back-button:hover {
        background: var(--light-green);
        color: var(--dark-green);
        transform: translateX(-5px);
    }

    .back-button::before {
        content: '←';
        transition: transform 0.3s ease;
    }

    .back-button:hover::before {
        transform: translateX(-5px);
    }

    /* Адаптивность */
    @media (max-width: 576px) {
        .tours-title {
            font-size: 2.2rem;
        }

        .header-actions {
            flex-direction: column;
            align-items: stretch;
        }

        .create-tour-btn {
            justify-content: center;
        }

        .header-stats {
            justify-content: center;
        }

        .tour-content {
            padding: 20px;
        }

        .tour-actions {
            flex-direction: column;
        }

        .filters-section {
            padding: 20px;
        }

        .filter-form {
            grid-template-columns: 1fr;
        }
    }

    /* Прогресс загрузки */
    .upload-progress {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(255, 255, 255, 0.9);
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s ease;
    }

    .upload-progress.active {
        opacity: 1;
        visibility: visible;
    }

    .progress-spinner {
        width: 50px;
        height: 50px;
        border: 4px solid var(--light-green);
        border-top-color: var(--primary-green);
        border-radius: 50%;
        animation: spin 1s linear infinite;
        margin-bottom: 20px;
    }

    @keyframes spin {
        to { transform: rotate(360deg); }
    }
</style>

<div class="tours-page">
    <!-- Анимированный фон -->
    <div class="tours-background">
        <div class="tours-shape"></div>
        <div class="tours-shape"></div>
        <div class="tours-shape"></div>
    </div>

    <div class="container tours-container">
        <!-- Кнопка назад -->
        <a href="<?= \yii\helpers\Url::to(['admin/index']) ?>" class="back-button reveal">
            Назад в админ-панель
        </a>

        <!-- Заголовок -->
        <div class="tours-header reveal">
            <h1 class="tours-title">Управление турами</h1>

            <div class="header-actions">
                <?= Html::a('+ Создать новый тур', ['create-tour'], ['class' => 'create-tour-btn']) ?>

                <div class="header-stats">
                    <div class="stat-badge">
                        <span class="count"><?= count($tours) ?></span>
                        <span>Всего туров</span>
                    </div>
                    <div class="stat-badge">
    <span class="count">
        <?= number_format(
            round(
                array_reduce($tours, function($sum, $tour) {
                    return $sum + $tour->price;
                }, 0) / max(count($tours), 1)
            ),
            0,
            '',
            ' '
        ) ?>
    </span>
                        <span>Средняя цена</span>
                    </div>

                    <div class="stat-badge">
                        <span class="count"><?= array_reduce($tours, function($sum, $tour) { return $sum + ($tour->image ? 1 : 0); }, 0) ?></span>
                        <span>С фото</span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Фильтры -->


        <!-- Сетка туров -->
        <div class="tours-grid-section reveal">
            <div class="tours-grid" id="toursGrid">
                <?php if (empty($tours)): ?>
                    <div class="empty-state">
                        <div class="empty-icon">🏔️</div>
                        <h3>Туры не найдены</h3>
                        <p>Вы еще не создали ни одного тура. Начните с добавления первого тура!</p>
                        <?= Html::a('+ Создать первый тур', ['create-tour'], ['class' => 'create-tour-btn']) ?>
                    </div>
                <?php else: ?>
                    <?php foreach ($tours as $tour): ?>
                        <div class="tour-card" data-title="<?= Html::encode(mb_strtolower($tour->title)) ?>"
                             data-price="<?= $tour->price ?>" data-id="<?= $tour->id ?>">

                            <!-- Изображение тура -->
                            <div class="tour-image-container">
                                <?php if ($tour->image): ?>
                                    <img src="<?= Html::encode($tour->getImageUrl()) ?>"
                                         class="tour-image"
                                         alt="<?= Html::encode($tour->title) ?>">
                                <?php else: ?>
                                    <div style="
                                        width: 100%;
                                        height: 100%;
                                        background: linear-gradient(45deg, var(--light-green), var(--white));
                                        display: flex;
                                        align-items: center;
                                        justify-content: center;
                                        color: var(--gray);
                                    ">
                                        <span style="font-size: 3rem;">🏔️</span>
                                    </div>
                                <?php endif; ?>
                                <div class="tour-badge">
                                    <?= number_format($tour->price, 0, '', ' ') ?> ₽
                                </div>
                            </div>

                            <!-- Контент тура -->
                            <div class="tour-content">
                                <div class="tour-header">
                                    <h3 class="tour-title"><?= Html::encode($tour->title) ?></h3>
                                </div>

                                <div class="tour-description">
                                    <?= Html::encode($tour->description) ?>
                                </div>



                                <div class="tour-program">
                                    <div class="program-title">Программа тура</div>
                                    <div class="program-text">
                                        <?= Html::encode($tour->program) ?>
                                    </div>
                                </div>

                                <div class="tour-actions">

                                    <?= Html::a('✏️ Редактировать', ['update-tour', 'id' => $tour->id], [
                                        'class' => 'action-btn btn-edit',
                                        'title' => 'Редактировать тур'
                                    ]) ?>

                                    <?= Html::a('🗑️ Удалить', ['delete-tour', 'id' => $tour->id], [
                                        'class' => 'action-btn btn-delete',
                                        'title' => 'Удалить тур',
                                        'data-method' => 'post',
                                        'onclick' => 'return confirmDelete(this);'
                                    ]) ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Модальное окно подтверждения -->
        <div class="confirm-modal" id="confirmModal">
            <div class="confirm-content">
                <div class="confirm-icon">🗑️</div>
                <div class="confirm-text" id="confirmText">
                    Вы уверены, что хотите удалить этот тур? Это действие нельзя отменить.
                </div>
                <div class="confirm-actions">
                    <button class="confirm-btn cancel" onclick="closeConfirmModal()">Отмена</button>
                    <button class="confirm-btn delete" id="confirmDeleteBtn">Удалить</button>
                </div>
            </div>
        </div>

        <!-- Прогресс загрузки -->
        <div class="upload-progress" id="uploadProgress">
            <div class="progress-spinner"></div>
            <p style="color: var(--dark-green); font-weight: 600;">Загрузка...</p>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Инициализация анимаций
        const reveals = document.querySelectorAll('.reveal');

        function reveal() {
            reveals.forEach(element => {
                const windowHeight = window.innerHeight;
                const elementTop = element.getBoundingClientRect().top;
                const elementVisible = 100;

                if (elementTop < windowHeight - elementVisible) {
                    element.classList.add('active');
                }
            });
        }

        reveal();
        window.addEventListener('scroll', reveal);

        // Параллакс эффект для фона
        window.addEventListener('scroll', function() {
            const scrolled = window.pageYOffset;
            const background = document.querySelector('.tours-background');
            if (background) {
                background.style.transform = `translateY(${scrolled * 0.05}px)`;
            }
        });

        // Функции управления
        window.applyFilters = function() {
            const searchInput = document.getElementById('searchInput').value.toLowerCase();
            const sortSelect = document.getElementById('sortSelect').value;
            const priceFrom = document.getElementById('priceFrom').value;

            const cards = document.querySelectorAll('.tour-card');
            let visibleCards = [];

            // Применяем фильтры
            cards.forEach(card => {
                const title = card.getAttribute('data-title');
                const price = parseInt(card.getAttribute('data-price'));

                let visible = true;

                // Фильтр по поиску
                if (searchInput && !title.includes(searchInput.toLowerCase())) {
                    visible = false;
                }

                // Фильтр по цене
                if (priceFrom && price < parseInt(priceFrom)) {
                    visible = false;
                }

                if (visible) {
                    card.style.display = 'block';
                    visibleCards.push(card);
                    setTimeout(() => {
                        card.style.opacity = '1';
                        card.style.transform = 'translateY(0)';
                    }, 10);
                } else {
                    card.style.opacity = '0';
                    card.style.transform = 'translateY(20px)';
                    setTimeout(() => {
                        card.style.display = 'none';
                    }, 300);
                }
            });

            // Сортировка
            setTimeout(() => {
                sortCards(visibleCards, sortSelect);
            }, 350);

            // Показываем пустое состояние если нет карточек
            const grid = document.getElementById('toursGrid');
            const emptyState = grid.querySelector('.empty-state');
            const hasVisibleCards = Array.from(cards).some(card => card.style.display !== 'none');

            if (!hasVisibleCards && !emptyState) {
                const emptyStateHTML = `
                    <div class="empty-state">
                        <div class="empty-icon">🔍</div>
                        <h3>Туры не найдены</h3>
                        <p>Попробуйте изменить параметры поиска</p>
                        <button class="create-tour-btn" onclick="resetFilters()">
                            🔄 Сбросить фильтры
                        </button>
                    </div>
                `;
                grid.innerHTML = emptyStateHTML;
            } else if (hasVisibleCards && emptyState) {
                emptyState.remove();
            }
        };

        // Функция сортировки
        function sortCards(cards, sortBy) {
            const grid = document.getElementById('toursGrid');
            const sortedCards = Array.from(cards).sort((a, b) => {
                switch(sortBy) {
                    case 'title_asc':
                        return a.getAttribute('data-title').localeCompare(b.getAttribute('data-title'));
                    case 'title_desc':
                        return b.getAttribute('data-title').localeCompare(a.getAttribute('data-title'));
                    case 'price_asc':
                        return parseInt(a.getAttribute('data-price')) - parseInt(b.getAttribute('data-price'));
                    case 'price_desc':
                        return parseInt(b.getAttribute('data-price')) - parseInt(a.getAttribute('data-price'));
                    default:
                        return 0;
                }
            });

            // Перестраиваем сетку
            sortedCards.forEach(card => {
                grid.appendChild(card);

                // Анимация перемещения
                card.style.transition = 'transform 0.5s ease';
                setTimeout(() => {
                    card.style.transform = 'translateY(0)';
                }, 10);
            });
        }

        // Сброс фильтров
        window.resetFilters = function() {
            document.getElementById('searchInput').value = '';
            document.getElementById('sortSelect').value = 'title_asc';
            document.getElementById('priceFrom').value = '';

            const cards = document.querySelectorAll('.tour-card');
            cards.forEach(card => {
                card.style.display = 'block';
                setTimeout(() => {
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                }, 10);
            });

            // Удаляем пустое состояние если есть
            const emptyState = document.querySelector('.empty-state');
            if (emptyState) {
                emptyState.remove();
            }
        };

        // Подтверждение удаления
        let deleteUrl = '';

        window.confirmDelete = function(element) {
            event.preventDefault();
            deleteUrl = element.href;

            const modal = document.getElementById('confirmModal');
            modal.classList.add('active');

            // Блокируем прокрутку
            document.body.style.overflow = 'hidden';

            return false;
        };

        window.closeConfirmModal = function() {
            const modal = document.getElementById('confirmModal');
            modal.classList.remove('active');

            // Разблокируем прокрутку
            document.body.style.overflow = '';
        };

        // Обработчик удаления
        document.getElementById('confirmDeleteBtn').addEventListener('click', function() {
            const progress = document.getElementById('uploadProgress');
            progress.classList.add('active');

            // Симуляция удаления
            setTimeout(() => {
                progress.classList.remove('active');
                closeConfirmModal();

                // В реальном приложении здесь будет AJAX запрос
                // window.location.href = deleteUrl;

                // Демо: удаляем карточку из DOM
                const cardToRemove = document.querySelector(`.tour-card[data-id="${deleteUrl.split('id=')[1]}"]`);
                if (cardToRemove) {
                    cardToRemove.style.opacity = '0';
                    cardToRemove.style.transform = 'scale(0.8) translateY(20px)';
                    setTimeout(() => {
                        cardToRemove.remove();

                        // Обновляем статистику
                        updateStats();

                        // Показываем уведомление
                        showNotification('Тур удален', 'Тур успешно удален из системы', 'success');
                    }, 300);
                }
            }, 1500);
        });

        // Функция обновления статистики
        function updateStats() {
            const totalTours = document.querySelectorAll('.tour-card').length;
            const stats = document.querySelectorAll('.stat-badge .count');

            if (stats[0]) {
                stats[0].textContent = totalTours;
            }

            // Анимация обновления
            stats.forEach(stat => {
                stat.style.transform = 'scale(1.2)';
                setTimeout(() => {
                    stat.style.transform = 'scale(1)';
                }, 300);
            });
        }

        // Функция показа уведомлений
        function showNotification(title, message, type = 'success') {
            const notification = document.createElement('div');
            notification.className = 'notification';
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: white;
                padding: 15px 25px;
                border-radius: 10px;
                box-shadow: 0 10px 30px rgba(0,0,0,0.15);
                border-left: 4px solid var(--${type === 'success' ? 'primary-green' : 'error'});
                z-index: 1000;
                animation: slideInRight 0.3s ease;
                max-width: 350px;
            `;

            notification.innerHTML = `
                <strong>${title}</strong><br>
                <small>${message}</small>
            `;

            document.body.appendChild(notification);

            // Автоматическое удаление через 5 секунд
            setTimeout(() => {
                notification.style.opacity = '0';
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 5000);
        }

        // Добавляем CSS для анимации уведомлений
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideInRight {
                from {
                    transform: translateX(100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
        `;
        document.head.appendChild(style);

        // Поиск по мере ввода
        const searchInput = document.getElementById('searchInput');
        let searchTimeout;

        searchInput.addEventListener('input', function() {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(() => {
                applyFilters();
            }, 500);
        });

        // Эффекты для карточек при наведении
        document.querySelectorAll('.tour-card').forEach(card => {
            card.addEventListener('mouseenter', function() {
                this.style.zIndex = '10';
            });

            card.addEventListener('mouseleave', function() {
                this.style.zIndex = '';
            });
        });

        // Автоматическое обновление (демо)
        setInterval(() => {
            // Симулируем обновление данных
            const badges = document.querySelectorAll('.tour-badge');
            badges.forEach(badge => {
                const currentPrice = parseInt(badge.textContent.replace(/[^0-9]/g, ''));
                if (Math.random() > 0.9) {
                    const newPrice = currentPrice + Math.floor(Math.random() * 1000);
                    badge.textContent = newPrice.toLocaleString('ru-RU') + ' ₽';
                    badge.style.animation = 'none';
                    setTimeout(() => {
                        badge.style.animation = 'pulse 0.5s ease';
                    }, 10);
                }
            });
        }, 10000);

        // Добавляем CSS для анимации пульсации
        const pulseStyle = document.createElement('style');
        pulseStyle.textContent = `
            @keyframes pulse {
                0% { transform: scale(1); }
                50% { transform: scale(1.1); }
                100% { transform: scale(1); }
            }
        `;
        document.head.appendChild(pulseStyle);

        // Ленивая загрузка изображений
        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    if (img.dataset.src) {
                        img.src = img.dataset.src;
                        img.removeAttribute('data-src');
                    }
                    observer.unobserve(img);
                }
            });
        }, {
            rootMargin: '50px 0px',
            threshold: 0.1
        });

        // Применяем ленивую загрузку ко всем изображениям
        document.querySelectorAll('.tour-image').forEach(img => {
            if (img.src) {
                img.dataset.src = img.src;
                img.src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAwIiBoZWlnaHQ9IjMwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMTAwJSIgaGVpZ2h0PSIxMDAlIiBmaWxsPSIjZjVmNWY1Ii8+PC9zdmc+';
                imageObserver.observe(img);
            }
        });
    });
</script>