<?php
use yii\helpers\Html;

$this->title = 'Панель администратора: заявки';
?>

<style>
    :root {
        --primary-green: #4CAF7D;
        --light-green: #E8F5E9;
        --accent-green: #81C784;
        --dark-green: #388E3C;
        --white: #FFFFFF;
        --gray-light: #F5F5F5;
        --gray: #757575;
        --gray-dark: #424242;
        --warning: #FF9800;
        --error: #f44336;
        --blue: #2196F3;
        --purple: #9C27B0;
        --cyan: #00BCD4;
    }

    .bookings-page {
        min-height: 100vh;
        background: linear-gradient(135deg, var(--light-green) 0%, var(--white) 100%);
        padding: 40px 0;
        position: relative;
        overflow: hidden;
    }

    /* Анимированный фон */
    .bookings-background {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        z-index: 0;
        opacity: 0.1;
    }

    .bookings-shape {
        position: absolute;
        border-radius: 50%;
        background: var(--primary-green);
        filter: blur(60px);
        opacity: 0.3;
        animation: float 20s infinite ease-in-out;
    }

    .bookings-shape:nth-child(1) {
        width: 400px;
        height: 400px;
        top: -100px;
        right: -100px;
        animation-delay: 0s;
    }

    .bookings-shape:nth-child(2) {
        width: 300px;
        height: 300px;
        bottom: -50px;
        left: -50px;
        animation-delay: -5s;
    }

    @keyframes float {
        0%, 100% {
            transform: translate(0, 0) rotate(0deg);
        }
        33% {
            transform: translate(30px, 30px) rotate(120deg);
        }
        66% {
            transform: translate(-20px, 40px) rotate(240deg);
        }
    }

    /* Основной контейнер */
    .bookings-container {
        position: relative;
        z-index: 1;
        max-width: 1400px;
        margin: 0 auto;
        padding: 0 20px;
    }

    /* Заголовок */
    .bookings-header {
        text-align: center;
        margin-bottom: 50px;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease forwards;
    }

    .bookings-title {
        font-size: 3rem;
        font-weight: 800;
        background: linear-gradient(45deg, var(--dark-green), var(--blue));
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        margin-bottom: 15px;
    }

    .stats-bar {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
        margin-top: 30px;
    }

    .stat-item {
        background: var(--white);
        padding: 20px;
        border-radius: 15px;
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.08);
        display: flex;
        align-items: center;
        gap: 15px;
        transition: all 0.3s ease;
    }

    .stat-item:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 35px rgba(0, 0, 0, 0.12);
    }

    .stat-icon {
        font-size: 2.5rem;
        width: 60px;
        height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
        background: rgba(76, 175, 125, 0.1);
        color: var(--primary-green);
    }

    .stat-content h3 {
        font-size: 1.8rem;
        color: var(--dark-green);
        margin: 0;
    }

    .stat-content p {
        color: var(--gray);
        margin: 5px 0 0;
        font-size: 0.9rem;
    }

    /* Фильтры */
    .filters-section {
        background: var(--white);
        border-radius: 20px;
        padding: 30px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
        margin-bottom: 40px;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.3s forwards;
    }

    .filters-title {
        font-size: 1.3rem;
        color: var(--dark-green);
        margin-bottom: 20px;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .filters-title::before {
        content: '🔍';
        font-size: 1.2rem;
    }

    .filter-form {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 20px;
        align-items: end;
    }

    .filter-group {
        flex: 1;
    }

    .filter-label {
        display: block;
        margin-bottom: 8px;
        color: var(--gray-dark);
        font-weight: 600;
        font-size: 0.95rem;
    }

    .filter-select {
        width: 100%;
        padding: 14px 20px;
        border: 2px solid var(--light-green);
        border-radius: 12px;
        font-size: 1rem;
        color: var(--gray-dark);
        background: var(--white);
        transition: all 0.3s ease;
        outline: none;
        cursor: pointer;
        appearance: none;
        background-image: url("data:image/svg+xml;charset=UTF-8,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none' stroke='%23757575' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3e%3cpolyline points='6 9 12 15 18 9'%3e%3c/polyline%3e%3c/svg%3e");
        background-repeat: no-repeat;
        background-position: right 20px center;
        background-size: 16px;
    }

    .filter-select:focus {
        border-color: var(--primary-green);
        box-shadow: 0 0 0 4px rgba(76, 175, 125, 0.1);
    }

    /* Таблица заявок */
    .bookings-table-section {
        background: var(--white);
        border-radius: 20px;
        padding: 40px;
        box-shadow: 0 15px 40px rgba(0, 0, 0, 0.1);
        overflow: hidden;
        opacity: 0;
        transform: translateY(30px);
        animation: fadeInUp 0.8s ease 0.6s forwards;
    }

    .table-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 30px;
        flex-wrap: wrap;
        gap: 20px;
    }

    .table-title {
        font-size: 1.8rem;
        color: var(--dark-green);
        margin: 0;
    }

    .table-controls {
        display: flex;
        gap: 15px;
        align-items: center;
    }

    .export-btn {
        padding: 12px 25px;
        background: var(--primary-green);
        color: var(--white);
        border: none;
        border-radius: 10px;
        cursor: pointer;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 8px;
        transition: all 0.3s ease;
        text-decoration: none;
        font-size: 0.95rem;
    }

    .export-btn:hover {
        background: var(--dark-green);
        transform: translateY(-2px);
    }

    .refresh-btn {
        padding: 12px;
        background: var(--light-green);
        border: none;
        border-radius: 10px;
        cursor: pointer;
        color: var(--dark-green);
        transition: all 0.3s ease;
    }

    .refresh-btn:hover {
        background: var(--primary-green);
        color: var(--white);
        transform: rotate(180deg);
    }

    /* Таблица */
    .bookings-table {
        width: 100%;
        border-collapse: separate;
        border-spacing: 0;
        margin: 0;
    }

    .bookings-table thead th {
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: var(--white);
        font-weight: 600;
        padding: 20px;
        text-align: left;
        border: none;
        position: sticky;
        top: 0;
        z-index: 10;
    }

    .bookings-table thead th:first-child {
        border-top-left-radius: 15px;
    }

    .bookings-table thead th:last-child {
        border-top-right-radius: 15px;
    }

    .bookings-table tbody tr {
        transition: all 0.3s ease;
        border-bottom: 1px solid var(--light-green);
    }

    .bookings-table tbody tr:hover {
        background: rgba(76, 175, 125, 0.05);
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
    }

    .bookings-table tbody td {
        padding: 20px;
        color: var(--gray-dark);
        border: none;
        vertical-align: middle;
    }

    .bookings-table tbody tr:last-child {
        border-bottom: none;
    }

    /* Статусы */
    .status-badge {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        padding: 8px 16px;
        border-radius: 50px;
        font-size: 0.9rem;
        font-weight: 600;
        white-space: nowrap;
    }

    .status-new {
        background: rgba(33, 150, 243, 0.1);
        color: var(--blue);
        border: 1px solid rgba(33, 150, 243, 0.2);
    }

    .status-review {
        background: rgba(255, 152, 0, 0.1);
        color: var(--warning);
        border: 1px solid rgba(255, 152, 0, 0.2);
    }

    .status-accepted {
        background: rgba(76, 175, 125, 0.1);
        color: var(--primary-green);
        border: 1px solid rgba(76, 175, 125, 0.2);
    }

    .status-rejected {
        background: rgba(244, 67, 54, 0.1);
        color: var(--error);
        border: 1px solid rgba(244, 67, 54, 0.2);
    }

    /* Действия */
    .action-buttons {
        display: flex;
        gap: 10px;
        flex-wrap: wrap;
    }

    .action-btn {
        padding: 10px 18px;
        border: none;
        border-radius: 8px;
        font-size: 0.85rem;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 6px;
        text-decoration: none;
        white-space: nowrap;
    }

    .action-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
    }

    .btn-warning {
        background: linear-gradient(45deg, var(--warning), #FFB74D);
        color: white;
    }

    .btn-success {
        background: linear-gradient(45deg, var(--primary-green), var(--accent-green));
        color: white;
    }

    .btn-info {
        background: linear-gradient(45deg, var(--blue), #64B5F6);
        color: white;
    }

    .btn-danger {
        background: linear-gradient(45deg, var(--error), #EF5350);
        color: white;
    }

    /* Пагинация */
    .pagination-section {
        margin-top: 40px;
        display: flex;
        justify-content: center;
        opacity: 0;
        transform: translateY(20px);
        animation: fadeInUp 0.8s ease 0.9s forwards;
    }

    .pagination {
        display: flex;
        gap: 8px;
        list-style: none;
        padding: 0;
        margin: 0;
    }

    .page-item {
        min-width: 40px;
    }

    .page-link {
        display: flex;
        align-items: center;
        justify-content: center;
        height: 40px;
        padding: 0 15px;
        background: var(--white);
        border: 2px solid var(--light-green);
        border-radius: 10px;
        color: var(--gray-dark);
        text-decoration: none;
        font-weight: 600;
        transition: all 0.3s ease;
    }

    .page-link:hover {
        background: var(--primary-green);
        color: var(--white);
        border-color: var(--primary-green);
        transform: translateY(-2px);
    }

    .page-item.active .page-link {
        background: var(--primary-green);
        color: var(--white);
        border-color: var(--primary-green);
    }

    /* Анимации */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(40px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    /* Кастомный скроллбар для таблицы */
    .table-wrapper {
        overflow-x: auto;
        border-radius: 15px;
        margin: 0 -10px;
        padding: 10px;
    }

    .table-wrapper::-webkit-scrollbar {
        height: 8px;
    }

    .table-wrapper::-webkit-scrollbar-track {
        background: var(--light-green);
        border-radius: 4px;
    }

    .table-wrapper::-webkit-scrollbar-thumb {
        background: var(--primary-green);
        border-radius: 4px;
    }

    .table-wrapper::-webkit-scrollbar-thumb:hover {
        background: var(--dark-green);
    }

    /* Модальное окно для деталей */
    .modal-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.5);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s ease;
    }

    .modal-overlay.active {
        opacity: 1;
        visibility: visible;
    }

    .modal-content {
        background: var(--white);
        border-radius: 20px;
        padding: 40px;
        max-width: 500px;
        width: 90%;
        max-height: 80vh;
        overflow-y: auto;
        transform: translateY(30px);
        transition: transform 0.3s ease;
    }

    .modal-overlay.active .modal-content {
        transform: translateY(0);
    }

    /* Адаптивность */
    @media (max-width: 1200px) {
        .bookings-container {
            padding: 0 15px;
        }

        .bookings-table-section {
            padding: 25px;
        }
    }

    @media (max-width: 768px) {
        .bookings-title {
            font-size: 2.2rem;
        }

        .table-header {
            flex-direction: column;
            align-items: stretch;
        }

        .action-buttons {
            flex-direction: column;
        }

        .action-btn {
            width: 100%;
            justify-content: center;
        }

        .stats-bar {
            grid-template-columns: 1fr;
        }

        .bookings-table {
            font-size: 0.9rem;
        }

        .bookings-table td,
        .bookings-table th {
            padding: 12px 8px;
        }
    }

    @media (max-width: 576px) {
        .bookings-table-section {
            padding: 20px;
        }

        .filters-section {
            padding: 20px;
        }

        .filter-form {
            grid-template-columns: 1fr;
        }

        .bookings-title {
            font-size: 1.8rem;
        }
    }

    /* Кнопка назад */
    .back-button {
        display: inline-flex;
        align-items: center;
        gap: 10px;
        padding: 12px 25px;
        background: var(--white);
        color: var(--gray-dark);
        text-decoration: none;
        border-radius: 50px;
        font-weight: 500;
        margin-bottom: 30px;
        transition: all 0.3s ease;
        box-shadow: 0 5px 15px rgba(76, 175, 125, 0.1);
    }

    .back-button:hover {
        background: var(--light-green);
        color: var(--dark-green);
        transform: translateX(-5px);
    }

    .back-button::before {
        content: '←';
        transition: transform 0.3s ease;
    }

    .back-button:hover::before {
        transform: translateX(-5px);
    }

    /* Пустое состояние */
    .empty-state {
        text-align: center;
        padding: 60px 20px;
        color: var(--gray);
    }

    .empty-icon {
        font-size: 4rem;
        margin-bottom: 20px;
        opacity: 0.5;
    }

    .empty-state h3 {
        color: var(--gray-dark);
        margin-bottom: 10px;
    }

    /* Загрузка */
    .loading {
        display: inline-block;
        width: 20px;
        height: 20px;
        border: 3px solid var(--light-green);
        border-top-color: var(--primary-green);
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }

    @keyframes spin {
        to { transform: rotate(360deg); }
    }
</style>

<div class="bookings-page">
    <!-- Анимированный фон -->
    <div class="bookings-background">
        <div class="bookings-shape"></div>
        <div class="bookings-shape"></div>
    </div>

    <div class="container bookings-container">
        <!-- Кнопка назад -->
        <a href="<?= \yii\helpers\Url::to(['admin/index']) ?>" class="back-button reveal">
            Назад в админ-панель
        </a>

        <!-- Заголовок -->
        <div class="bookings-header reveal">
            <h1 class="bookings-title">Управление заявками</h1>

            <div class="stats-bar">
                <div class="stat-item">
                    <div class="stat-icon">📋</div>
                    <div class="stat-content">
                        <h3 id="total-bookings"><?= count($bookings) ?></h3>
                        <p>Всего заявок</p>
                    </div>
                </div>

                <div class="stat-item">
                    <div class="stat-icon">🆕</div>
                    <div class="stat-content">
                        <h3 id="new-bookings">0</h3>
                        <p>Новых заявок</p>
                    </div>
                </div>

                <div class="stat-item">
                    <div class="stat-icon">⏳</div>
                    <div class="stat-content">
                        <h3 id="review-bookings">0</h3>
                        <p>На рассмотрении</p>
                    </div>
                </div>

                <div class="stat-item">
                    <div class="stat-icon">✅</div>
                    <div class="stat-content">
                        <h3 id="accepted-bookings">0</h3>
                        <p>Принято</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Фильтры -->
        <div class="filters-section reveal">
            <h3 class="filters-title">Фильтры и поиск</h3>

            <form method="get" class="filter-form" id="filterForm">
                <div class="filter-group">
                    <label class="filter-label">Статус заявки</label>
                    <select name="status" class="filter-select" onchange="document.getElementById('filterForm').submit()">
                        <option value="">Все статусы</option>
                        <option value="Новая" <?= $status == 'Новая' ? 'selected' : '' ?>>Новая</option>
                        <option value="На рассмотрении" <?= $status == 'На рассмотрении' ? 'selected' : '' ?>>На рассмотрении</option>
                        <option value="Заявка принята" <?= $status == 'Заявка принята' ? 'selected' : '' ?>>Принята</option>
                    </select>
                </div>





                <div class="filter-group">
                    <button type="button" class="export-btn" onclick="resetFilters()">
                        <span>🔄</span> Сбросить фильтры
                    </button>
                </div>
            </form>
        </div>

        <!-- Таблица заявок -->
        <div class="bookings-table-section reveal">
            <div class="table-header">
                <h2 class="table-title">Список заявок</h2>
                <div class="table-controls">

                </div>
            </div>

            <div class="table-wrapper">
                <table class="bookings-table">
                    <thead>
                    <tr>
                        <th>Пользователь</th>
                        <th>Тур</th>
                        <th>Дата тура</th>
                        <th>Участники</th>
                        <th>Статус</th>
                        <th>Действия</th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php if (empty($bookings)): ?>
                        <tr>
                            <td colspan="6" class="empty-state">
                                <div class="empty-icon">📭</div>
                                <h3>Заявки не найдены</h3>
                                <p>Попробуйте изменить параметры фильтрации</p>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach($bookings as $b): ?>
                            <tr data-id="<?= $b->id ?>" data-status="<?= $b->status ?>">
                                <td>
                                    <strong><?= Html::encode($b->user->full_name) ?></strong><br>
                                    <small style="color: var(--gray);">📧 <?= Html::encode($b->user->email) ?></small><br>
                                    <small style="color: var(--gray);">📱 <?= Html::encode($b->user->phone) ?></small>
                                </td>
                                <td>
                                    <strong><?= Html::encode($b->tour->title) ?></strong><br>
                                    <small style="color: var(--gray);">💰 <?= number_format($b->tour->price, 0, '', ' ') ?> ₽</small>
                                </td>
                                <td>
                                    <strong><?= date('d.m.Y', strtotime($b->tour_date)) ?></strong><br>
                                    <small style="color: var(--gray);">📅 Создана: <?= date('d.m.Y', strtotime($b->created_at)) ?></small>
                                </td>
                                <td>
                                    <div class="persons-badge" style="
                                            background: var(--light-green);
                                            color: var(--dark-green);
                                            padding: 8px 16px;
                                            border-radius: 20px;
                                            font-weight: bold;
                                            display: inline-block;
                                        ">
                                        👥 <?= Html::encode($b->persons) ?> чел.
                                    </div><br>
                                    <small style="color: var(--gray);">💵 <?= number_format($b->persons * $b->tour->price, 0, '', ' ') ?> ₽</small>
                                </td>
                                <td>
                                    <?php
                                    $statusClass = '';
                                    if ($b->status == 'Новая') $statusClass = 'status-new';
                                    elseif ($b->status == 'На рассмотрении') $statusClass = 'status-review';
                                    elseif ($b->status == 'Заявка принята') $statusClass = 'status-accepted';
                                    ?>
                                    <div class="status-badge <?= $statusClass ?>">
                                        <?php if ($b->status == 'Новая'): ?>🆕
                                        <?php elseif ($b->status == 'На рассмотрении'): ?>⏳
                                        <?php elseif ($b->status == 'Заявка принята'): ?>✅
                                        <?php endif; ?>
                                        <?= Html::encode($b->status) ?>
                                    </div>
                                </td>
                                <td>
                                        <?php if ($b->status == 'Новая'): ?>
                                            <?= Html::a('⏳ Рассмотрение', ['update-status', 'id' => $b->id, 'status' => 'На рассмотрении'], [
                                                'class' => 'action-btn btn-warning',
                                                'title' => 'Перевести в рассмотрение'
                                            ]) ?>
                                        <?php endif; ?>

                                        <?php if (in_array($b->status, ['Новая', 'На рассмотрении'])): ?>
                                            <?= Html::a('✅ Принять', ['update-status', 'id' => $b->id, 'status' => 'Заявка принята'], [
                                                'class' => 'action-btn btn-success',
                                                'title' => 'Принять заявку'
                                            ]) ?>
                                        <?php endif; ?>

                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Пагинация (если есть) -->
            <?php if (isset($pagination)): ?>
                <div class="pagination-section">
                    <ul class="pagination">
                        <?php if ($pagination->page > 0): ?>
                            <li class="page-item">
                                <a class="page-link" href="<?= \yii\helpers\Url::current(['page' => $pagination->page]) ?>">
                                    ← Назад
                                </a>
                            </li>
                        <?php endif; ?>

                        <?php for ($i = 1; $i <= $pagination->pageCount; $i++): ?>
                            <li class="page-item <?= ($i == $pagination->page + 1) ? 'active' : '' ?>">
                                <a class="page-link" href="<?= \yii\helpers\Url::current(['page' => $i - 1]) ?>">
                                    <?= $i ?>
                                </a>
                            </li>
                        <?php endfor; ?>

                        <?php if ($pagination->page < $pagination->pageCount - 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="<?= \yii\helpers\Url::current(['page' => $pagination->page + 2]) ?>">
                                    Вперед →
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Инициализация анимаций
        const reveals = document.querySelectorAll('.reveal');

        function reveal() {
            reveals.forEach(element => {
                const windowHeight = window.innerHeight;
                const elementTop = element.getBoundingClientRect().top;
                const elementVisible = 100;

                if (elementTop < windowHeight - elementVisible) {
                    element.classList.add('active');
                }
            });
        }

        reveal();
        window.addEventListener('scroll', reveal);

        // Подсчет статистики
        function calculateStats() {
            const rows = document.querySelectorAll('.bookings-table tbody tr[data-status]');
            let newCount = 0;
            let reviewCount = 0;
            let acceptedCount = 0;
            let totalCount = rows.length;

            rows.forEach(row => {
                const status = row.getAttribute('data-status');
                switch(status) {
                    case 'Новая': newCount++; break;
                    case 'На рассмотрении': reviewCount++; break;
                    case 'Заявка принята': acceptedCount++; break;
                }
            });

            // Анимируем изменение чисел
            animateCounter('new-bookings', newCount);
            animateCounter('review-bookings', reviewCount);
            animateCounter('accepted-bookings', acceptedCount);
        }

        // Анимация счетчика
        function animateCounter(elementId, targetValue) {
            const element = document.getElementById(elementId);
            if (!element) return;

            let current = parseInt(element.textContent) || 0;
            const increment = targetValue > current ? 1 : -1;
            const stepTime = Math.abs(50 / (targetValue - current));

            function updateCounter() {
                current += increment;
                element.textContent = current;

                if (current !== targetValue) {
                    setTimeout(updateCounter, stepTime);
                }
            }

            updateCounter();
        }

        // Запускаем подсчет статистики
        calculateStats();

        // Эффекты для строк таблицы
        document.querySelectorAll('.bookings-table tbody tr').forEach(row => {
            row.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-2px)';
            });

            row.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
            });

            // Клик для просмотра деталей
            row.addEventListener('click', function(e) {
                if (!e.target.closest('.action-btn')) {
                    const id = this.getAttribute('data-id');
                    if (id) {
                        window.location.href = `<?= \yii\helpers\Url::to(['admin/view']) ?>?id=${id}`;
                    }
                }
            });
        });

        // Параллакс эффект для фона
        window.addEventListener('scroll', function() {
            const scrolled = window.pageYOffset;
            const background = document.querySelector('.bookings-background');
            if (background) {
                background.style.transform = `translateY(${scrolled * 0.05}px)`;
            }
        });

        // Функции управления
        window.resetFilters = function() {
            window.location.href = '<?= \yii\helpers\Url::to(['admin/index']) ?>';
        };

        window.exportToExcel = function() {
            alert('Экспорт в Excel начат. В реальном приложении здесь будет экспорт данных.');
            // В реальном приложении здесь будет вызов API для экспорта
        };

        // Подсветка выбранного фильтра
        const currentFilter = '<?= $status ?>';
        if (currentFilter) {
            const select = document.querySelector('select[name="status"]');
            if (select) {
                select.style.borderColor = 'var(--primary-green)';
                select.style.boxShadow = '0 0 0 3px rgba(76, 175, 125, 0.1)';
            }
        }

        // Автоматическое обновление статистики (демо)
        setInterval(() => {
            // Симуляция обновления статистики
            const newCountEl = document.getElementById('new-bookings');
            if (newCountEl) {
                const current = parseInt(newCountEl.textContent);
                const randomChange = Math.random() > 0.7 ? 1 : 0;
                if (randomChange > 0) {
                    animateCounter('new-bookings', current + randomChange);
                    animateCounter('total-bookings', <?= count($bookings) ?> + randomChange);

                    // Показываем уведомление
                    showNotification('Новая заявка', 'Поступила новая заявка на тур', 'info');
                }
            }
        }, 30000); // Каждые 30 секунд

        // Функция показа уведомлений
        function showNotification(title, message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = 'notification';
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: white;
                padding: 15px 25px;
                border-radius: 10px;
                box-shadow: 0 10px 30px rgba(0,0,0,0.15);
                border-left: 4px solid var(--${type === 'info' ? 'primary-green' : type});
                z-index: 1000;
                animation: slideInRight 0.3s ease;
            `;

            notification.innerHTML = `
                <strong>${title}</strong><br>
                <small>${message}</small>
            `;

            document.body.appendChild(notification);

            // Автоматическое удаление через 5 секунд
            setTimeout(() => {
                notification.style.opacity = '0';
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 5000);
        }

        // Добавляем CSS для анимации уведомлений
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideInRight {
                from {
                    transform: translateX(100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
        `;
        document.head.appendChild(style);

        // Сортировка таблицы (демо)
        document.querySelectorAll('.bookings-table th').forEach((th, index) => {
            th.style.cursor = 'pointer';
            th.addEventListener('click', function() {
                const table = this.closest('table');
                const tbody = table.querySelector('tbody');
                const rows = Array.from(tbody.querySelectorAll('tr'));

                // Сортировка строк
                rows.sort((a, b) => {
                    const aValue = a.querySelector(`td:nth-child(${index + 1})`).textContent;
                    const bValue = b.querySelector(`td:nth-child(${index + 1})`).textContent;

                    if (index === 3) { // Для числовых значений
                        const aNum = parseInt(aValue.match(/\d+/)?.[0] || 0);
                        const bNum = parseInt(bValue.match(/\d+/)?.[0] || 0);
                        return aNum - bNum;
                    } else if (index === 4) { // Для статусов
                        const statusOrder = { 'Новая': 1, 'На рассмотрении': 2, 'Заявка принята': 3, 'Отклонена': 4 };
                        return statusOrder[aValue] - statusOrder[bValue];
                    }

                    return aValue.localeCompare(bValue);
                });

                // Пересобираем таблицу
                rows.forEach(row => tbody.appendChild(row));

                // Анимация
                rows.forEach((row, i) => {
                    setTimeout(() => {
                        row.style.opacity = '0.5';
                        row.style.transform = 'translateY(10px)';
                        setTimeout(() => {
                            row.style.opacity = '1';
                            row.style.transform = 'translateY(0)';
                        }, 100);
                    }, i * 50);
                });
            });
        });
    });
</script>