<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use app\models\{Tour, RegisterForm, LoginForm, Booking, Review};

class SiteController extends Controller
{

    public function behaviors()
    {
        return [
            'access' => [
                'class' => \yii\filters\AccessControl::class,
                'only' => ['logout'],
                'rules' => [
                    [
                        'actions' => ['logout'],
                        'allow' => true,
                        'roles' => ['@'], // только авторизованные
                    ],
                ],
            ],
            'verbs' => [
                'class' => \yii\filters\VerbFilter::class,
                'actions' => [
                    'logout' => ['post'],
                ],
            ],
        ];
    }

    public function actionIndex()
    {
        // Получаем все туры
        $tours = \app\models\Tour::find()->all();

        // Получаем все отзывы с данными пользователей
        $reviews = \app\models\Review::find()->with('user')->all();

        return $this->render('index', [
            'tours' => $tours,
            'reviews' => $reviews,
        ]);
    }


    public function actionCatalog()
    {
        $request = Yii::$app->request->get('DynamicModel', []); // <- здесь берем массив DynamicModel
        $minPrice = $request['minPrice'] ?? null;
        $maxPrice = $request['maxPrice'] ?? null;

        $filterModel = new \yii\base\DynamicModel([
            'minPrice' => $minPrice,
            'maxPrice' => $maxPrice,
        ]);
        $filterModel->addRule(['minPrice', 'maxPrice'], 'safe');

        $query = \app\models\Tour::find();

        if ($minPrice !== null && $minPrice !== '') {
            $query->andWhere(['>=', 'price', $minPrice]);
        }
        if ($maxPrice !== null && $maxPrice !== '') {
            $query->andWhere(['<=', 'price', $maxPrice]);
        }

        $tours = $query->all();

        return $this->render('catalog', [
            'tours' => $tours,
            'filterModel' => $filterModel,
        ]);
    }


    public function actionLogout()
    {
        Yii::$app->user->logout();
        return $this->goHome(); // редирект на главную страницу
    }



    public function actionTour($id)
    {
        return $this->render('tour', ['tour' => Tour::findOne($id)]);
    }

    public function actionRegister()
    {
        $model = new RegisterForm();
        if ($model->load(Yii::$app->request->post()) && $model->register())
            return $this->redirect(['login']);
        return $this->render('register', ['model' => $model]);
    }

    public function actionLogin()
    {
        $model = new LoginForm();
        if ($model->load(Yii::$app->request->post()) && $model->login())
            return $this->redirect(['index']);
        return $this->render('login', ['model' => $model]);
    }

    public function actionBooking($id)
    {
        $tour = Tour::findOne($id);
        if (!$tour) throw new \yii\web\NotFoundHttpException('Тур не найден');

        $model = new \app\models\Booking();
        $model->user_id = Yii::$app->user->id;

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Заявка успешно создана!');
            return $this->redirect(['site/my-bookings']);
        }

        return $this->render('booking', ['model' => $model, 'tour' => $tour]);
    }


    public function actionMyBookings()
    {
        $userId = Yii::$app->user->id;

        // Получаем все заявки пользователя с туром
        $bookings = \app\models\Booking::find()
            ->where(['user_id' => $userId])
            ->with('tour', 'reviews') // подтягиваем отзывы тура
            ->all();

        $review = new \app\models\Review(); // модель для новой формы отзыва

        if ($review->load(Yii::$app->request->post()) && $review->save()) {
            Yii::$app->session->setFlash('success', 'Отзыв успешно добавлен!');
            return $this->refresh();
        }

        return $this->render('my-bookings', [
            'bookings' => $bookings,
            'review' => $review,
        ]);
    }

}
